/********************************* 

Bunching analysis for Charging Ahead
AEJ:Applied
See readme.txt file for further detail

*********************************/

********************************
* Preliminaries 
********************************

set more off
clear all
set matsize 1000
set seed 5925169

/* 
* cap cd // change user directory

* ssc install boottest // user written command for wild bootstrap

* Set macros

global resdir  // destination for tables and figures
global bootrep 500 // change to desired number 

* Read in analysis file

use "../workingdata_analysis.dta", clear

*/


********************************
* Descriptive figures 
********************************

// Figure 4, Panel a and Figure A.9 // 

twoway (hist mcons_no0 if mcons_no0 < 550 & mcons_no0 > 150 & bill_year > 2012 & switcher == 1 ///
	& switchmonthD == 0 & pp == 0 & lifeline == 1, width(10) fcolor(gs10) lcolor(gs11) frac) ///
	(hist mcons_no0 if mcons_no0 < 550 & mcons_no0 > 150 & bill_year > 2012 & switcher == 1 ///
	& switchmonthD == 0 & pp == 1 & lifeline == 1, width(10) fcolor(none) lcolor(black) frac) ///
	(scatter lifeline2_2014 mcons_no0 if mcons_no0 < 550 & mcons_no0 > 150, connect(stairstep) lcolor(gs12) mcolor(none) lpattern(dash_dot) lwidth(thick) yaxis(2)), ///
	xscale(range(150(50)550)) ytitle(Fraction) ytitle("Marginal price (USD)", axis(2)) yscale(range(0(.05).25) axis(2)) ///
	legend(order(1 "Postpaid" 2 "Prepaid")) graphregion(fcolor(white) lcolor(white)) ylabel(0(.05).25, axis(2)) ///
	xtitle(Total monthly usage) 
	graph export "$resdir/MP_bunch_ll.pdf", replace	

twoway (hist mcons_no0 if mcons_no0 < 800 & mcons_no > 400 & bill_year > 2012 & switcher == 1 ///
	& switchmonthD == 0 & pp == 0 & lifeline == 0, width(10) fcolor(gs10) lcolor(gs11) frac) ///
	(hist mcons_no0 if mcons_no0 < 800 & mcons_no > 400 & bill_year > 2012 & switcher == 1 ///
	& switchmonthD == 0 & pp == 1 & lifeline == 0, width(10) fcolor(none) lcolor(black) frac) ///
	(scatter domestic_2014 mcons_no0 if mcons_no0 < 800 & mcons_no > 400, connect(stairstep) lcolor(gs12) lpattern(dash_dot) lwidth(thick) mcolor(none) yaxis(2)), ///
	ytitle(Fraction) ytitle("Marginal price (USD)", axis(2))  yscale(range(0(.05).25) axis(2)) ylabel(0(.05).25, axis(2)) ///
	legend(order(1 "Postpaid" 2 "Prepaid")) graphregion(fcolor(white) lcolor(white)) ///
	xtitle(Total monthly usage)
	graph export "$resdir/MP_bunch_dom.pdf", replace	

twoway (hist mcons_no0 if mcons_no0 < 1000 & bill_year > 2012 & switcher == 1 ///
	& switchmonthD == 0 & pp == 0 & lifeline == 1, width(10) fcolor(gs10) lcolor(gs11) frac) ///
	(hist mcons_no0 if mcons_no0 < 1000 & bill_year > 2012 & switcher == 1 ///
	& switchmonthD == 0 & pp == 1 & lifeline == 1, width(10) fcolor(none) lcolor(black) frac) ///
	(scatter lifeline2_2014 mcons_no0 if mcons_no0 < 1000, connect(stairstep) lcolor(gs12) mcolor(none) lpattern(dash_dot) lwidth(thick) yaxis(2)), ///
	ytitle(Fraction) ytitle("Marginal price (USD)", axis(2)) yscale(range(0(.05).25) axis(2)) ///
	legend(order(1 "Postpaid" 2 "Prepaid")) graphregion(fcolor(white) lcolor(white)) ylabel(0(.05).25, axis(2)) ///
	xtitle(Total monthly usage) 
	graph export "$resdir/MP_bunch_ll_full.pdf", replace	

twoway (hist mcons_no0 if mcons_no0 < 1000 & bill_year > 2012 & switcher == 1 ///
	& switchmonthD == 0 & pp == 0 & lifeline == 0, width(10) fcolor(gs10) lcolor(gs11) frac) ///
	(hist mcons_no0 if mcons_no0 < 1000 & bill_year > 2012 & switcher == 1 ///
	& switchmonthD == 0 & pp == 1 & lifeline == 0, width(10) fcolor(none) lcolor(black) frac) ///
	(scatter domestic_2014 mcons_no0 if mcons_no0 < 1000, connect(stairstep) lcolor(gs12) lpattern(dash_dot) lwidth(thick) mcolor(none) yaxis(2)), ///
	ytitle(Fraction) ytitle("Marginal price (USD)", axis(2))  yscale(range(0(.05).25) axis(2)) ylabel(0(.05).25, axis(2)) ///
	legend(order(1 "Postpaid" 2 "Prepaid")) graphregion(fcolor(white) lcolor(white)) ///
	xtitle(Total monthly usage)
	graph export "$resdir/MP_bunch_dom_full.pdf", replace


********************************
* Bunching analysis 
********************************

local data "MP"										// MP for phase in sample; elast for larger CoCT sample

local r 500 										// replications for bootstrapping SEs

if "`data'" == "MP" { 								// run both with 2013 / 0 and 2015 / 1 
	global y "2015"
	global m "1"
	}
	
	else {											
		global y "2013 2014 2015"
		global m "0 1"
		}

// define programs for block bootstrap: myboot_elast and myboot_mass

	* elasticity 
		cap program drop myboot_elast 
			program define myboot_elast, rclass																				
				preserve
				bsample
		
				qui reg freq bin_* if window == 0, cl(id)
				predict cfact
				label var cfact "counterfactual predicted distribution"
								
				egen totalcfact = total(cfact), mi
				egen totalactual = total(freq), mi
				
				sum totalcfact totalactual
				
				local i= 0.01
				local j= 1
				while (totalactual>totalcfact){
					local j = `j'+ `i'
					di "`j'"
					replace cfact = cfact*`j'
					drop totalcfact
					egen totalcfact = total(cfact), mi
					if (totalactual<=totalcfact) break
				}
					
				gen pred1 = .
				replace pred1 = cfact if window == 1		
				egen pred_mass = total(pred1), mi			
				label var pred_mass "counterfactual mass"
								
				gen prag1 = .
				replace prag1 = freq if window == 1
				egen act_mass = total(prag1), mi
				label var act_mass "actual mass"
					
				gen  bunch_mass = (act_mass-pred_mass)/pred_mass
				label var bunch_mass "bunching mass"
												
				gen elasticity = bunch_mass/((t2-t3)/t2) 
				sum elasticity 
				return scalar elast = r(mean)
				
				restore
			end	
			
		* mass
		
			cap program drop myboot_mass 
			
			program define myboot_mass, rclass																				
				preserve
				bsample
		
				qui reg freq bin_* if window == 0, cl(id)
				predict cfact
				label var cfact "counterfactual predicted distribution"
							
				egen totalcfact = total(cfact), mi
				egen totalactual = total(freq), mi
				
				sum totalcfact totalactual
				
				local i= 0.01
				local j= 1
				while (totalactual>totalcfact){
					local j = `j'+ `i'
					di "`j'"
					replace cfact = cfact*`j'
					drop totalcfact
					egen totalcfact = total(cfact), mi
					if (totalactual<=totalcfact) break
				}
					
				gen pred1 = .
				replace pred1 = cfact if window == 1		
				egen pred_mass = total(pred1), mi			
				label var pred_mass "counterfactual mass"
								
				gen prag1 = .
				replace prag1 = freq if window == 1
				egen act_mass = total(prag1), mi
				label var act_mass "actual mass"
					
				gen  bunch_mass = (act_mass-pred_mass)/pred_mass
				label var bunch_mass "bunching mass"
				sum bunch_mass										
				return scalar bmass = r(mean)						
				
				restore
			end	
			
// Data set up
			
	cap drop if switchmonthD == 1
	cap drop if switcher == 0
	cap drop group
	keep if tariff == 1 // Lifeline only
	keep if bill_year > 2012
	sum mcons_no0, d
	drop if mcons_no0 > `r(p99)' // trim very high outliers
	drop if mcons_no0 < 0 
	global maxcons = `r(p99)' 
	
	cap drop t2 t3
	gen t2 = 0.079 if bill_year == 2013
	replace t2 = 0.084 if bill_year == 2014
	replace t2 = 0.091 if bill_year == 2015
	gen t3 = 0.184 if bill_year == 2013
	replace t3 = 0.204 if bill_year == 2014
	replace t3 = 0.251 if bill_year == 2015


	cap file close elast		
	if "$m" == "0" { 
		file open elast using "$IN/Data/admin_data/elast.txt", write text replace
		
		file write elast "data" _tab "n" _tab "year" _tab "meter" _tab "window" _tab "poly" _tab "stat" _tab "observed" _tab "bootstrap" ///
			_tab "se" _tab "ci normal lb" _tab "ci normal ub" _tab _n	
	}
	
	else {											 
		file open elast using "$IN/Data/admin_data/elast.txt", write text append
		} 


// Loop over robustness checks 

	local bwidth = 10 

	foreach w in 2 3 4 5 6 {	

		foreach poly in 5 6 7 8 9 {
			preserve

				

// Analysis 

	egen bin = cut(mcons_no0), at(0(`bwidth')$maxcons)
	label var bin "consumption bin with width of `bwidth'"
	sum bin
	global num_bin = `r(max)'
	
	bys bin bill_year pp: gen freq = _N
	replace freq = 0 if freq == .
	label var freq "count of obs in a consumption bin by meter type and year"

	egen btag = tag(bin pp bill_year)
	gen delta = `w'*`bwidth'
	gen zstar = 350
	
		gen window = ((bin >= (zstar)) & (bin <= (zstar + delta))) // (bin >= (zstar - delta))
		label var window "within delta of kink point (symmetrical)" // trying all to the right
			
		forval i = 1/`poly' {					
			gen bin_`i' = bin ^ `i'
			}
		
		keep if btag == 1 

	tempfile bins
	save `bins', replace
	
	foreach y in $y { 
		foreach m in $m  {
		
			use `bins', clear
			
			keep if bill_year == `y' & pp == `m'			
			sum bin freq
			
				reg freq bin_* if window == 0, cl(id)
				predict cfact
				label var cfact "counterfactual predicted distribution"
			
				egen totalcfact = total(cfact), mi
				egen totalactual = total(freq), mi
				
				local i= 0.01
				local j= 1
				while (totalactual>totalcfact){
					local j = `j'+ `i'
					di "`j'"
					replace cfact = cfact*`j'
					drop totalcfact
					egen totalcfact = total(cfact), mi
					if (totalactual<=totalcfact) break
				}
		
				gen pred1 = .
				replace pred1 = cfact if window == 1
				egen pred_mass = total(pred1), mi
				label var pred_mass "counterfactual mass"
							
				gen prag1 = .
				replace prag1 = freq if window == 1
				egen act_mass = total(prag1), mi
				label var act_mass "actual mass"
				
				gen  bunch_mass = (act_mass-pred_mass)/pred_mass
				label var bunch_mass "bunching mass"
				sum bunch_mass 
				local observed_bmass = r(mean)
												
				gen elasticity = bunch_mass/((t2-t3)/t2) 
				sum elasticity 
	
				local observed_elast = r(mean)
				local n = r(N)
				
		// Figure 4, Panel B // 
		
				if `poly' == 7 & `w' == 4 {
					qui binscatter cfact freq bin, ///
					line(connect) xline(350) msymbol(none none) ///
					legend(order(1 "Counterfactual" 2 "Actual")) ///
					ytitle(Observations per bin) xtitle(Monthly kWh) 
					graph export "$resdir/elast_`y'_`m'.pdf", replace	
					}
				else {
					}

			sum *_mass		
			sum bunch_mass
			local bunch : display %-4.3f r(mean)
				
			* bootstrap elasticity se
						
			use `bins', clear
			keep if bill_year == `y' & pp == `m'
			
			simulate mean = r(elast) , reps(`r') seed(12345): myboot_elast

			bstat, stat(`observed_elast') n(`n')
			
			local b 	: display %-4.3f _b[mean]
			local se 	: display %-4.3f _se[mean]
			matrix bs_`y'_`m' = e(b_bs)
			local bs 	= bs_`y'_`m'[1,1]
			matrix ci_n_`y'_`m' = e(ci_normal)
			local ci_n_lb = ci_n_`y'_`m'[1,1]
			local ci_n_ub = ci_n_`y'_`m'[2,1]
			matrix ci_b_`y'_`m' = e(ci_bc)
			local ci_b_lb = ci_b_`y'_`m'[1,1]
			local ci_b_ub = ci_b_`y'_`m'[2,1]

			estat bootstrap, all
			 

		file write elast "`data'" _tab "`n'" _tab "`y'" _tab "`m'" _tab "`w'" _tab "`poly'" _tab "elast" _tab "`b'" _tab "`bs'" _tab ///
			"`se'" _tab "`ci_b_lb'" _tab "`ci_b_ub'" _n	
			
			*bootstrap mass se
			di "mass bootstrap"
	
			use `bins', clear
			keep if bill_year == `y' & pp == `m'
			
			simulate mean = r(bmass) , reps(`r') seed(12345): myboot_mass

			bstat, stat(`observed_bmass') n(`n')
			
			local b 	: display %-4.3f _b[mean]
			local se 	: display %-4.3f _se[mean]
			matrix bs_`y'_`m' = e(b_bs)
			local bs 	= bs_`y'_`m'[1,1]
			matrix ci_n_`y'_`m' = e(ci_normal)
			local ci_n_lb = ci_n_`y'_`m'[1,1]
			local ci_n_ub = ci_n_`y'_`m'[2,1]
			matrix ci_b_`y'_`m' = e(ci_bc)
			local ci_b_lb = ci_b_`y'_`m'[1,1]
			local ci_b_ub = ci_b_`y'_`m'[2,1]

			estat bootstrap, all

		file write elast "`data'" _tab "`n'" _tab "`y'" _tab "`m'" _tab "`w'" _tab "`poly'" _tab "mass" _tab "`b'" _tab "`bs'" _tab ///
			"`se'" _tab "`ci_b_lb'" _tab "`ci_b_ub'" _n				
		
			}
		}
		
			restore
			} 
		
		} 
		
		file close elast
		
	
// Table B.1 //

	import delimited "$IN/Data/admin_data/elast.txt", varnames(1) clear

	gen t_elast = .
	gen t_mass = .
	
	cap file close bunching
		file open bunching using ///
			"$resdir/MP_bunching.tex", ///
			write text replace
		file write bunching "\begin{tabular}{cccccccc}"	_n
		file write bunching "\hline" _n
		file write bunching "& &  \multicolumn{3}{c}{Excess mass} & \multicolumn{3}{c}{Elasticity} \\" _n
		file write bunching " Window & Polynomial & Postpaid & Prepaid & T-stat & Postpaid & Prepaid & T-stat \\ " _n
		file write bunching " & & (1) & (2) & (3) & (4) & (5) & (6) \\ \hline" _n

	foreach w in 2 3 4 5 6 {		

		foreach poly in 5 7 9 {
			
			
			foreach stat in elast mass {
			
				sum bootstrap if stat == "`stat'" & year == 2013 & poly == `poly' & window == `w'
				local b1_`stat'  : display %-4.3f r(mean)
				sum se if stat == "`stat'" & year == 2013 & poly == `poly' & window == `w'
				local se1_`stat'  : display %-4.3f r(mean)
				sum n if year == 2013 & poly == `poly' & window == `w'
				local n1  : display %-4.3f r(mean)
	
				sum bootstrap if stat == "`stat'" & year == 2015 & poly == `poly' & window == `w'
				local b2_`stat'  : display %-4.3f r(mean)
				sum se if stat == "`stat'" & year == 2015 & poly == `poly' & window == `w'
				local se2_`stat'  : display %-4.3f r(mean)
				sum n if year == 2015 & poly == `poly' & window == `w'
				local n2  : display %-4.3f r(mean)
				
			replace t_`stat' = (`b2_`stat''-`b1_`stat'') / (( ((((`se2_`stat'')^2)*`n2')/(`n2')) + ((((`se1_`stat'')^2)*`n1')/(`n1')) )^0.5) if poly == `poly' & window == `w'
				sum t_`stat'
				local t_`stat' : display %-4.3f r(mean)
				
				}
			
			local window = `w'*10
		
		file write bunching " `window' & `poly' & `b1_mass' & `b2_mass' & `t_mass' & `b1_elast' & `b2_elast' & `t_elast' \\ " _n
		file write bunching "  &  & (`se1_mass') & (`se2_mass') &  & (`se1_elast') & (`se2_elast') &  \\ " _n

				
			}
		}
		
		
		file write bunching "\hline" _n
		file write bunching "\end{tabular}"
		file close bunching	
