###################################################################
# Name              : FirstStageForIVInteracted_toyprice_foodprice.R
# Description       : This program estimates of the investment functions 
#                     (first stage) and reduced form equations with 
#                     toy price and food price as instruments corresponding 
#                     to a production function with interactions between 
#                     all variables and the treatment dummy. 
#                     This means a non-interacted first stage, but a fully 
#                     interacted reduced form. 
###################################################################




###################################################################
# Define function to estimate first stage and reduced form functions by OLS
# This function produces "naive" estimates not correcting for the 
# fact that we're using factor scores as well as adjusted 
# estimates correcting for that fact 
###################################################################

first.rdform.estim.func             <- function(scores, data, eps, mean, cov, prob, missing){
  
  # Load data
  setwd(dir_data)     
  measures <- read.csv(data, header=T)
  measures <- measures[order(measures$treat), ]
  n        <- nrow(measures)
  
  # Get estimates of measurement system Ineed for estimation of correction bias 
  totalmean    <- matrix(0, nF, nG)
  for (g in 1:nG)  totalmean[,g] <- prob[[g]] %*% mean[[g]]
  
  # Define observables 
  Zall         <- cbind(measures$meanprice_juglibros_log_st, measures$meanprice_juglibros_log_st*measures$treat, 
                        measures$FUmeanprice_log_st,  measures$FUmeanprice_log_st*measures$treat, 
                        measures$tpobc_pop1993_st,  measures$tpobc_pop1993_st*measures$treat)
  X1           <- cbind(measures$ln_nkids0_st, measures$ln_nkids0_st*measures$treat) 
  
  
  mscoresall   <- rbind(scores[[1]], scores[[2]])
  keep         <- which(complete.cases(cbind(Zall,X1, mscoresall)))
  keep1        <- which(complete.cases(cbind(Zall,X1, mscoresall)[which(measures$treat==1),]))
  keep2        <- which(complete.cases(cbind(Zall,X1, mscoresall)[which(measures$treat==2),]))
  treatnomiss  <- measures$treat[keep]  
  
  if (missing=="FALSE"){
    fs   <- list()
    fs[[1]] <- scores[[1]][keep1,]
    fs[[2]] <- scores[[2]][keep2,]
  } else if (missing=="TRUE"){
    fs   <- scores
  }
  
  pos           <- c(2,4,7,8)
  
  # First stage (not interacted, using toy price and food price as instruments) 
  Z1          <-  cbind(measures$meanprice_juglibros_log_st, measures$FUmeanprice_log_st)
  Znames      <-  c("toyprice", "foodprice")
  
  X1           <- cbind(measures$ln_nkids0_st) 
  Xnames       <- c("kids")
  
  Z           <-  Z1[keep,]
  X           <-  X1[keep,]
  
  RHSnames    <- c("treat", "cog0", "ncog0", "cogmo", "ncogmo", Xnames, Znames)
  
  out.first.stage   <-  first.stage.func(fs, X, Xnames, Z, Znames, treatnomiss, RHSnames, pos, totalmean, mean, cov, prob, parametric=1)
  

  # Reduced form with interactions between all variables and treatment dummy
  Z1          <-  cbind(measures$meanprice_juglibros_log_st, measures$meanprice_juglibros_log_st*measures$treat, 
                        measures$FUmeanprice_log_st,  measures$FUmeanprice_log_st*measures$treat)
  Znames     <-  c("toyprice","toyprice_T", "foodprice", "foodprice_T")
  X1           <- cbind(measures$ln_nkids0_st, measures$ln_nkids0_st*measures$treat) 
  Xnames       <- c("kids","kids_T")
  
  Z           <-  Z1[keep,]
  X           <-  X1[keep,]
  
  RHSnames    <- c("treat", "cog0", "ncog0", "cogmo", "ncogmo", "cog0_T", "ncog0_T", "cogmo_T", "ncogmo_T", Xnames, Znames)
  
  out.rdform        <-  rdform.interacttreat.func(fs, X, Xnames, Z, Znames, treatnomiss, RHSnames, pos, totalmean, mean, cov, prob, parametric=1)
  
  
  regdata           <- data.frame(cbind(rbind(fs[[1]],fs[[2]]), rbind(fs[[1]],fs[[2]]) *  treatnomiss ,X, Z, treatnomiss))
  colnames(regdata) <- c("cog1", "cog0", "ncog1", "ncog0", "mat", "time", "cogmo", "ncogmo", 
                         "cog1_T", "cog0_T", "ncog1_T", "ncog0_T", "mat_T", "time_T", "cogmo_T", "ncogmo_T", Xnames, Znames, "treat")
  return(list(out.first.stage, out.rdform, regdata))
} 




###################################################################
# Estimate first stage and reduced form equations on true data and 
# bootstrapped samples using function above 
###################################################################
for (boot in 0:Bootstrap){
  
  ## Estimate on true data 
  if (boot==0){  
    # Load input needed for function 
    setwd(dir_outputFM)
    load("trueFM.R")
    load("fs_true.R")
    
    # Estimate first stage and reduced forms using FS estimates obtained with BFGS estimator
    out_true           <- first.rdform.estim.func(list(fs_true[[5]],fs_true[[6]]), "measures.csv", 
                                                  eps, mean.mix, cov.mix, prob.mix, missing="FALSE")
    
    # Rename output 
    invest_true       <- out_true[[1]][[1]]
    investbias_true   <- out_true[[1]][[2]]
    cf_true           <- out_true[[1]][[3]]
    
    rdform_true       <- out_true[[2]][[1]]
    rdformbias_true   <- out_true[[2]][[2]]
    
    regdata_true      <- out_true[[3]]
  } 
  
  ## Estimate on bootstrapped data
  if (boot==1 ){ 
    noflagFS <- noflag
    nof <- length(noflagFS)
    setwd(dir_outputFM)
    load("fs_bstrap.R")
    
    # Define output 
    invest_bstrap <- array(0, dim=c(nrow(invest_true), ncol(invest_true), nof))
    investbias_bstrap <- array(0, dim=c(nrow(investbias_true), ncol(investbias_true), nof))
    cf_bstrap    <- list()
    
    rdform_bstrap <- array(0, dim=c(nrow(rdform_true), ncol(rdform_true), nof))
    rdformbias_bstrap <- array(0, dim=c(nrow(rdformbias_true), ncol(rdformbias_true), nof))
    
    
    for (b in 1:nof){
      # Estimate first stage and reduced forms 
      out_bstrap   <- first.rdform.estim.func(fs_bstrap_allBFGS[[noflagFS[b]]], paste("measures_b", noflagFS[b], ".csv", sep=""), 
                                              epsBoot[,noflagFS[b]],
                                              list(meanBoot[,,noflagFS[b],1], meanBoot[,,noflagFS[b],2]), 
                                              list(covBoot[,,noflagFS[b],1],covBoot[,,noflagFS[b],2]), 
                                              list(probBoot[,noflagFS[b],1], probBoot[,noflagFS[b],2]), 
                                              missing="FALSE")
      
      print(b)
      # Rename output 
      invest_bstrap[,,b]        <- out_bstrap[[1]][[1]]
      investbias_bstrap[,,b]    <- out_bstrap[[1]][[2]]
      cf_bstrap[[b]]            <- out_bstrap[[1]][[3]]
      
      rdform_bstrap[,,b]        <- out_bstrap[[2]][[1]]
      rdformbias_bstrap[,,b]    <- out_bstrap[[2]][[2]]
      
      regdata                    <- out_bstrap[[3]]
      save(regdata, file=paste("regdata_bstrap", b, ".R", sep=""))
      
    } 
    
  } 
  
} 

###################################################################
# Compute optimal weighting matrix for minimum distance estimator 
###################################################################
rdform_cog        <- rdform_bstrap[,1,] 
rdform_ncog       <- rdform_bstrap[,2,] 
cogrdf_var        <- 1/(nof - 1) * (rdform_cog  - rowMeans(rdform_cog)) %*% t(rdform_cog  - rowMeans(rdform_cog))
ncogrdf_var       <- 1/(nof - 1) * (rdform_ncog - rowMeans(rdform_ncog)) %*% t(rdform_ncog  - rowMeans(rdform_ncog))




##############################################################################
# Create table of estimates (point estimate followed by bootstrapped 
# standard errors, 95% ci, p-value one tail test and p-value two tailed test)
############################################################################## 

outtrue     <- list(invest_true, rdform_true)  
outb        <- list(invest_bstrap, rdform_bstrap) 

outtable        <- list()
for (s in 1:2) {
  npar                 <- nrow(outtrue[[s]])
  outtable[[s]]        <- matrix(0, npar*5, ncol(outtrue[[s]]))
  
  for (i in 1:npar){
    for (j in 1:ncol(outtrue[[s]])){
      
      
      mean             <- mean(outb[[s]][i,j,])
      se               <- sd(outb[[s]][i,j,])
      t_stat           <- mean/se
      t_crit           <- (outb[[s]][i,j,] - mean)/se
      p_val            <- 1 - ecdf(t_crit)(t_stat)
      p_val2           <- 1 - ecdf(t_crit^2)(t_stat^2)
      
      outtable[[s]][(i*5-4), j]  <- round(mean,3)
      outtable[[s]][(i*5-3),j]   <- round(se,3) 
      outtable[[s]][(i*5-2), j]  <- paste("[", round(quantile(outb[[s]][i,j,], .025),3), ",", 
                                          round(quantile(outb[[s]][i,j,], .975),3), "]", sep="") 
      outtable[[s]][(i*5-1), j]  <- p_val 
      outtable[[s]][i*5, j]      <- p_val2 
      
    } 
  } 
} 


# Add ftest and cragg donal tests to investment table
outtable[[1]]   <- rbind(outtable[[1]])

# Name output rows and columns 
basic    <- c("Intercept", "", "","","",
              "Treat", "", "", "","",
              "cog0","", "", "","",
              "ncog0", "", "", "","",
              "cogmo", "", "", "","",
              "ncogmo", "", "", "","") 

Xnames      <- c( "kids")
Znames     <-  c("toyprice", "foodprice")
invnames <- basic 
for (l in 1:length(c(Xnames, Znames))) invnames     <- c(invnames, c(Xnames, Znames)[l] , "", "","", "")


Xnames      <- c( "kids", "kids_T")
Znames     <-  c("toyprice","toyprice_T", "foodprice", "foodprice_T")
rdfnames  <- c(basic, "cog0_T","", "", "","",
               "ncog0_T", "", "", "","",
               "cogmo_T", "", "", "","",
               "ncogmo_T", "", "", "","") 
for (l in 1:length(c(Xnames, Znames))) rdfnames     <- c(rdfnames, c(Xnames, Znames)[l] , "", "","", "")


rownames(outtable[[1]]) <- invnames 
rownames(outtable[[2]]) <- rdfnames
colnames(outtable[[2]]) <- c("Cognitive skills", "Socio-emotional skills")
colnames(outtable[[1]]) <- c("Material investment", "Time investment")


# Save output 
setwd(dir_outputPF)
save(invest_true, rdform_true, invest_bstrap,  rdform_bstrap, file=paste("OUTPUT", ".R", sep=""))
write.csv(outtable[[1]], file=paste("INVEST",  ".csv", sep=""))
write.csv(cbind(outtable[[2]]), file=paste("RDF",  ".csv", sep=""))

