** Program for AJKM 2019 (Age and High-Growth Entrepreneurship)


/* Table of Contents

==== Main Manuscript =====
Figure 1: Founder Age Distribution: All Startups and High Growth Startups
Figure 2: Likelihood of Extreme Success, Conditional on Starting a Firm
Table 1: Founder Age - Avg across US and by Technology Definition
Table 2: Founder Age and Success - Upper Tail Growth or Acquisition
Table 3: Industry-Specific Experience and Growth Outcomes

==== Online Appendix =====
Figure A1: Founder Rates by Age
Figure A2: Results by Founder Definition and Legal Form
Figure A3: Forward Stock Multiples as the Founder Ages: Apple, MSFT, Amazon, Google
Figure A4: All Startups and High Performance Startups by Sales
Figure A5: Founder Age Distribution and Founders' Prior Wages
Figure A6: Wage Distributions of Non-Founders, Founders, and Successful Founders

Table A1: Founder Age - Perceptions from Media & Two Prominent VCs
Table A3: Founder Age - Oldest and Youngest Tech Sectors
Table A4: Mean Founder Age by Calendear Year of Firm's Founding
Table A5: Minimum and Maximum Ages within Founder Teams

*/



program define setup
use "/projects9/dms-p1-988-hc/project45/output/k1_founders_2007_2014", clear
append using "/projects9/dms-p1-988-hc/project45/output/ccorp_founders_2007_2014"
gen k1 = (k1_flag == 1) /* 1 if K-1 (S-corporation, Partnerships) firm, 0 if C-corporation*/
gen exit = (acq == 1 | ipo == 1)

gen group1 = 1 /*All*/
gen group2 = (state_num == 6) /*California*/
gen group3 = (state_num == 25) /*Mass*/
gen group4 = (state_num == 36) /*New York */
gen group5 = (sv_zip == 1) /* Silicon Valley zipcodes */
gen group6 = (hub_zip2007 == 1) /* Hub regions defined by Guzman and Stern (2016) */

gen naics4 = substr(bestnaics,1,4)
destring naics4, replace

end


*==========================================================================
* Figure 1: Founder Age Distribution: All Startups and High Growth Startups
*===========================================================================

setup
keep if age >= 20 & age <= 65
bysort age: egen count_all = sum(1)
bysort age: egen count_top1_abs5 = sum(top1_abs5)
egen total_count = sum(1)
egen total_top1_abs = sum(top1_abs5)
gen perc_all = count_all / total_count
gen perc_top1_abs5 = count_top1_abs5 / total_top1_abs
bysort age: keep if _n == 1

twoway (scatter perc_all age) ///
	(scatter perc_top1_abs5 age) 
	
	
*==========================================================================
* Figure 2: Likelihood of Extreme Success, Conditional on Starting a Firm
*	.. figure 2a: exit (acquired or IPO)
*	.. figure 2b: top 0.1% employment growth
*===========================================================================

setup
foreach var in exit topx1_abs5 {
preserve
reg `var' i.age i.year, r
parmest, saving("plot_`var'.dta",replace) idstr(`var')
use plot_`var', clear
keep in 1/46
gen age = _n + 19
serrbar estimate stderr age, scale(1.96)
graph save Graph $outreg/homerun_plot_`var', replace
restore
}



*==========================================================================
* Table 1: Founder Age - Avg across US and by Technology Definition
*	.. loop through All, High-Tech, VC-backed, Patent owning ACROSS:
*		.. All, California, Mass, NY, Silicon Valley, Hub
*===========================================================================

setup
gen group1 = 1 /*All*/
gen group2 = (state_num == 6) /*California*/
gen group3 = (state_num == 25) /*Mass*/
gen group4 = (state_num == 36) /*New York */
gen group5 = (sv_zip == 1) /* Silicon Valley zipcodes */
gen group6 = (hub_zip2007 == 1) /* Hub regions defined by Guzman and Stern (2016) */

foreach X of numlist 1/6 {

display "--------------- (1) ALL STARTUPS ---------------"
preserve
keep if group`X' == 1
summ age
restore

display "--------------- (2) HIGH TECH STARTUPS ---------------"
preserve
keep if high_tech_all == 1 & group`X' == 1
summ age
restore

display "--------------- (3) VC-BACKED STARTUPS ---------------"
preserve
keep if vc_ever == 1 & group`X' == 1
summ age
restore

display "--------------- (4) PATENT-OWNING STARTUPS ---------------"
preserve
keep if pat_ever == 1 & group`X' == 1
summ age
restore

}



*==========================================================================
* Table 2: Founder Age and Success - Upper Tail Growth or Acquisition
*	.. loop through All, top 10%, top 5%, top1%, top0.1%, Exit ACROSS:
*		.. High-Tech, VC-Backed, Patent-owning firms
*==========================================================================

foreach X of numlist 1/6 {
preserve
display "--------------- (1) ALL STARTUPS ---------------"
keep if group`X' == 1 & year < 2010
	summ age
restore

preserve
display "--------------- (2) Top 10% growth (5-yr) ---------------"
keep if group`X' == 1 & top10_abs5 == 1
	summ age
restore

preserve
display "--------------- (3) Top 5% growth (5-yr) ---------------"
keep if group`X' == 1 & top5_abs5 == 1
	summ age
restore

preserve
display "--------------- (4) Top 1% growth (5-yr) ---------------"
keep if group`X' == 1 & top1_abs5 == 1
	summ age
restore

preserve
display "--------------- (5) Top 0.1% growth (5-yr) ---------------"
keep if group`X' == 1 & topx1_abs5 == 1
	summ age
restore

preserve
display "--------------- (6) Successful exit (IPO or acq) ---------------"
keep if group`X' == 1 & exit == 1
	summ age
restore
}


*==========================================================================
* Table 3: Industry-Specific Experience and Growth Outcomes
*	.. Separately at NAICS-2, NAICS-4, and NAICS-6 levels
*==========================================================================

use prior_founder_naics, clear

** NAICS-2 level analysis
foreach outcome in top10_abs5 top5_abs5 top1_abs5 topx1_abs5 exit {
	preserve
	display "=========== `var' with 0 years of NAICS-2 exp =============="
	keep if `outcome' == 1 & years_naics2 == 0
	summ age
	restore
}

foreach outcome in top10_abs5 top5_abs5 top1_abs5 topx1_abs5 exit {
	preserve
	display "=========== `var' with 1-2 years of NAICS-2 exp =============="
	keep if `outcome' == 1 & (years_naics2 == 1 | years_naics2 == 2)
	summ age
	restore
}

foreach outcome in top10_abs5 top5_abs5 top1_abs5 topx1_abs5 exit {
	preserve
	display "=========== `var' with 3+ years of NAICS-2 exp =============="
	keep if `outcome' == 1 & years_naics2 > 2
	summ age
	restore
}


** NAICS-4 level analysis
foreach outcome in top10_abs5 top5_abs5 top1_abs5 topx1_abs5 exit {
	preserve
	display "=========== `var' with 0 years of NAICS-4 exp =============="
	keep if `outcome' == 1 & years_naics4 == 0
	summ age
	restore
}

foreach outcome in top10_abs5 top5_abs5 top1_abs5 topx1_abs5 exit {
	preserve
	display "=========== `var' with 1-2 years of NAICS-4 exp =============="
	keep if `outcome' == 1 & (years_naics4 == 1 | years_naics4 == 2)
	summ age
	restore
}

foreach outcome in top10_abs5 top5_abs5 top1_abs5 topx1_abs5 exit {
	preserve
	display "=========== `var' with 3+ years of NAICS-4 exp =============="
	keep if `outcome' == 1 & years_naics4 > 2
	summ age
	restore
}


** NAICS-6 level analysis
foreach outcome in top10_abs5 top5_abs5 top1_abs5 topx1_abs5 exit {
	preserve
	display "=========== `var' with 0 years of NAICS-6 exp =============="
	keep if `outcome' == 1 & years_naics6 == 0
	summ age
	restore
}

foreach outcome in top10_abs5 top5_abs5 top1_abs5 topx1_abs5 exit {
	preserve
	display "=========== `var' with 1-2 years of NAICS-6 exp =============="
	keep if `outcome' == 1 & (years_naics6 == 1 | years_naics6 == 2)
	summ age
	restore
}

foreach outcome in top10_abs5 top5_abs5 top1_abs5 topx1_abs5 exit {
	preserve
	display "=========== `var' with 3+ years of NAICS-6 exp =============="
	keep if `outcome' == 1 & years_naics6 > 2
	summ age
	restore
}


*==========================================================================
* Figure A1: Founder Rates by Age
*	(a) plot of w2 wage-earner count by age
*	(b) rate of entr by age
*	(c) rate of entr by age in high tech sectors
*==========================================================================

qui setup

gen non_ht = (high_tech_all == 0)

/* approach: aggregate sums (and rates) pooling cohorts */
bysort age: egen entr_all = sum(1)
bysort age: egen entr_ht = sum(high_tech_all)
bysort age: egen entr_non_ht = sum(non_ht)
keep year age entr_all entr_ht entr_non_ht
bysort year age: keep if _n == 1

merge 1:1 year age using w2_age_2007_2014
keep if _merge == 3
drop _merge

bysort age: egen denom = sum(age_count)
drop age_count
bysort age: keep if _n == 1

foreach var in all ht {
gen rate_`var' = entr_`var' / denom
gen rate_norm_`var' = rate_`var' * 1000
gen count_norm_`var' = entr_`var' / 1000000
}
list age denom entr_all rate_norm_all entr_ht rate_norm_ht

** results
gen denom = age_count / 1000000
bysort age: keep if _n == 1
keep if age >= 20 & age <= 65
list age denom count_norm_all rate_norm_all count_norm_ht rate_norm_ht

graph twoway scatter pop_norm age /* figure A2a */
graph twoway scatter rate_norm_all age /* figure A2b */
graph twoway scatter rate_norm_ht age /* figure A2c */


*==========================================================================
* Figure A2: Results by Founder Definition and Legal Form
*	.. (1) ASE (C-corp) Owner-Workers
*	.. (2) K-1 Owner-Workers
*	.. (3) C-corp Top 3 earners
*	.. (4) K-1 Top 3 earners
*==========================================================================

** Graph 1: ASE C-corp Owner-Workers
use "/projects9/dms-p1-988-hc/project45/output/ase_ccorp_adjusted", clear
gen all = 1 if f_firstyear <= 2012
keep if founder == 1

foreach var in all top10_abs3 top5_abs3 top1_abs3 {
	display "====== Age bucket 1 for [subgroup] `var' ======="
	summ age if age_bucket_est == 1 & `var' == 1 & f_firstyear <= 2012
	
	display "====== Age bucket 2 for [subgroup] `var' ======="
	summ age if age_bucket_est == 2 & `var' == 1 & f_firstyear <= 2012
	
	display "====== Age bucket 3 for [subgroup] `var' ======="
	summ age if age_bucket_est == 3 & `var' == 1 & f_firstyear <= 2012
	
	display "====== Age bucket 4 for [subgroup] `var' ======="
	summ age if age_bucket_est == 4 & `var' == 1 & f_firstyear <= 2012
	
	display "====== Age bucket 5 for [subgroup] `var' ======="
	summ age if age_bucket_est == 5 & `var' == 1 & f_firstyear <= 2012
	
	display "====== Age bucket 6 for [subgroup] `var' ======="
	summ age if age_bucket_est == 6 & `var' == 1 & f_firstyear <= 2012
}


** Graph 2: K-1 Owner-Workers
use "/projects9/dms-p1-988-hc/project45/output/k1_founders_2007_2014", clear
gen all = 1 if year <= 2011

gen age_bucket =.
replace age_bucket = 1 if age < 25
replace age_bucket = 2 if age_bucket ==. & age <= 34
replace age_bucket = 3 if age_bucket ==. & age <= 44
replace age_bucket = 4 if age_bucket ==. & age <= 54
replace age_bucket = 5 if age_bucket ==. & age <= 64
replace age_bucket = 6 if age >= 65 & age ~=.

foreach var in all top10_abs3 top5_abs3 top1_abs3 {
	display "====== Age bucket 1 for [subgroup] `var' ======="
	summ age if age_bucket == 1 & `var' == 1
	
	display "====== Age bucket 2 for [subgroup] `var' ======="
	summ age if age_bucket == 2 & `var' == 1
	
	display "====== Age bucket 3 for [subgroup] `var' ======="
	summ age if age_bucket == 3 & `var' == 1
	
	display "====== Age bucket 4 for [subgroup] `var' ======="
	summ age if age_bucket == 4 & `var' == 1
	
	display "====== Age bucket 5 for [subgroup] `var' ======="
	summ age if age_bucket == 5 & `var' == 1
	
	display "====== Age bucket 6 for [subgroup] `var' ======="
	summ age if age_bucket == 6 & `var' == 1
}


** Graph 3: C-corp top 3 earners
use "/projects9/dms-p1-988-hc/project45/output/ccorp_founders_2007_2014", clear
gen all = 1 if year <= 2011

gen age_bucket =.
replace age_bucket = 1 if age < 25
replace age_bucket = 2 if age_bucket ==. & age <= 34
replace age_bucket = 3 if age_bucket ==. & age <= 44
replace age_bucket = 4 if age_bucket ==. & age <= 54
replace age_bucket = 5 if age_bucket ==. & age <= 64
replace age_bucket = 6 if age >= 65 & age ~=.

foreach var in all top10_abs3 top5_abs3 top1_abs3 topx1_abs3 {
	display "====== Age bucket 1 for [subgroup] `var' ======="
	summ age if age_bucket == 1 & `var' == 1
	
	display "====== Age bucket 2 for [subgroup] `var' ======="
	summ age if age_bucket == 2 & `var' == 1
	
	display "====== Age bucket 3 for [subgroup] `var' ======="
	summ age if age_bucket == 3 & `var' == 1
	
	display "====== Age bucket 4 for [subgroup] `var' ======="
	summ age if age_bucket == 4 & `var' == 1
	
	display "====== Age bucket 5 for [subgroup] `var' ======="
	summ age if age_bucket == 5 & `var' == 1
	
	display "====== Age bucket 6 for [subgroup] `var' ======="
	summ age if age_bucket == 6 & `var' == 1
}


** Graph 4: K-1 top 3 earners
use "/projects9/dms-p1-988-hc/project45/output/k1_w2_2007_2014", clear
gen all = 1 if year <= 2011

* Within-firm income: Highest 3 earners AND lowest earner
bysort ein year: egen team_size = sum(1)
bysort ein year: egen rank_earn = rank(-earn_ann), unique
gen earn_top3 = (rank_earn <= 3)
keep if earn_top3 == 1

gen age_bucket =.
replace age_bucket = 1 if age < 25
replace age_bucket = 2 if age_bucket ==. & age <= 34
replace age_bucket = 3 if age_bucket ==. & age <= 44
replace age_bucket = 4 if age_bucket ==. & age <= 54
replace age_bucket = 5 if age_bucket ==. & age <= 64
replace age_bucket = 6 if age >= 65 & age ~=.

foreach var in all top10_abs3 top5_abs3 top1_abs3 topx1_abs3 {
	display "====== Age bucket 1 for [subgroup] `var' ======="
	summ age if age_bucket == 1 & `var' == 1
	
	display "====== Age bucket 2 for [subgroup] `var' ======="
	summ age if age_bucket == 2 & `var' == 1
	
	display "====== Age bucket 3 for [subgroup] `var' ======="
	summ age if age_bucket == 3 & `var' == 1
	
	display "====== Age bucket 4 for [subgroup] `var' ======="
	summ age if age_bucket == 4 & `var' == 1
	
	display "====== Age bucket 5 for [subgroup] `var' ======="
	summ age if age_bucket == 5 & `var' == 1
	
	display "====== Age bucket 6 for [subgroup] `var' ======="
	summ age if age_bucket == 6 & `var' == 1
}


*==========================================================================
* Figure A3: Forward Stock Multiples as the Founder Ages: Apple, MSFT, Amazon, Google
*      This figure uses publicly available data.
*==========================================================================

** Apple
use Apple, clear
g jan1 = 1 if substr(date,1,2)=="1/"
g year = substr(date,-4,.)
keep if jan1==1
destring year, replace
sort year
g m5 = adjclose[_n+5]/adjclose
* Age of Steve Jobs
g age = year - 1955
twoway line m5 age if age<=56, ytitle("Forward 5 Year Stock Price Multiple") xtitle("Age") xline(21) title("Apple, Steve Jobs") xlabel(20(4)56) graphregion(color(white)) ylabel(0(5)20, angle(horizontal) format(%15.0f) tlength(.75)) 

** Amazon
use Amazon, clear
g jan1 = 1 if substr(date,1,2)=="1/"
g year = substr(date,-4,.)
keep if jan1==1
destring year, replace
sort year
g m5 = adjclose[_n+5]/adjclose
* Age of Jeff Bezos
g age = year - 1964
twoway line m5 age if age<=50, ytitle("Forward 5 Year Stock Price Multiple") xtitle("Age") xline(30) title("Amazon, Jeff Bezos") xlab(30(4)50) graphregion(color(white)) ylabel(0(2)6, angle(horizontal) format(%15.0f) tlength(.75)) 

** Microsoft
use Microsoft, clear
g jan1 = 1 if substr(date,1,2)=="1/"
g year = substr(date,-4,.)
keep if jan1==1
destring year, replace
sort year
g m5 = adjclose[_n+5]/adjclose
* Age of Bill Gates
g age = year - 1955
twoway line m5 age if age<=56, ytitle("Forward 5 Year Stock Price Multiple") xtitle("Age") xline(20) title("Microsoft, Bill Gates") xlab(20(4)56) graphregion(color(white)) ylabel(0(5)20, angle(horizontal) format(%15.0f) tlength(.75)) 

** Google
use Google, clear
g jan1 = 1 if substr(date,1,2)=="1/"
g year = substr(date,-4,.)
keep if jan1==1
destring year, replace
sort year
g m5 = adjclose[_n+5]/adjclose
* Age of Brin and Page (both born in 1973)
g age = year - 1973
twoway line m5 age if age<=44, ytitle("Forward 5 Year Stock Price Multiple") xtitle("Age") xline(25) title("Google, Brin and Page") xlab(24(4)44) graphregion(color(white)) ylabel(1(0.5)3.5, angle(horizontal) format(%15.0f) tlength(.75)) 



*==========================================================================
* Figure A4: All Startups and High Performance Startups by Sales
*==========================================================================

qui setup /* then join in top 1% revenues data */
merge m:1 ein year using "/projects9/dms-p1-988-hc/project45/output/top_revenues_ein", keepusing(top1_rev5)
drop if _merge == 2
drop _merge

keep if age >= 20 & age <= 65
bysort age: egen count_all = sum(1)
bysort age: egen count_top1_rev5 = sum(top1_rev5)
egen total_count = sum(1)
egen total_top1_rev = sum(top1_rev5)
gen perc_all = count_all / total_count
gen perc_top1_rev5 = count_top1_rev5 / total_top1_rev
bysort age: keep if _n == 1

twoway (scatter perc_all age) ///
	(scatter perc_top1_rev5 age) 
	
	
*==========================================================================
* Figure A5: Founder Age Distribution and Founders' Prior Wages
*==========================================================================

use "/projects9/dms-p1-988-hc/project45/output/founders_stack_2007_2014", clear /* Start with founder set */
gen exit = (acq == 1 | ipo == 1)
rename ein founding_ein
rename year founding_yr
merge 1:1 pik founding_ein using prior_founder_wage
keep if _merge == 3
drop _merge
drop year_prior
tempfile founders
save `founders'

use using prior_w2_worker_wage, clear /* Stack W2 Wage Workers (1% Random Sample) */
append using `founders', gen(founder)
gen nonfounder = (founder==0)

gen naics4_prior = substr(naics_prior,1,4)
destring naics4_prior, replace

gen earn_ann_prior_real=. /* Adjusting to $2010 real dollars */
replace earn_ann_prior_real = earn_ann_prior /	0.895460798	if year_w2 ==	2005
replace earn_ann_prior_real = earn_ann_prior /	0.92434663	if year_w2 ==	2006
replace earn_ann_prior_real = earn_ann_prior /	0.950481431	if year_w2 ==	2007
replace earn_ann_prior_real = earn_ann_prior /	0.987161852	if year_w2 ==	2008
replace earn_ann_prior_real = earn_ann_prior /	0.98349381	if year_w2 ==	2009
replace earn_ann_prior_real = earn_ann_prior /	1		if year_w2 ==	2010
replace earn_ann_prior_real = earn_ann_prior /	1.031178359	if year_w2 ==	2011
replace earn_ann_prior_real = earn_ann_prior /	1.052728106	if year_w2 ==	2012
replace earn_ann_prior_real = earn_ann_prior /	1.068317286	if year_w2 ==	2013
replace earn_ann_prior_real = earn_ann_prior /	1.085281981	if year_w2 ==	2014
gen ln_earn_ann = ln(earn_ann_prior+1)

qui reghdfe founder ib21.age, absorb(year_w2 naics4_prior) vce(robust)
outreg2 using $outreg/reg_age_entry.xls, append stats(coef se) dec(4) sideway noaster noparen
parmest, saving("$outreg/est_entry_age.dta",replace)

qui reghdfe founder ib21.age ln_earn_ann, absorb(year_w2 naics4_prior) vce(robust)
outreg2 using $outreg/reg_age_entry.xls, append stats(coef se) dec(4) sideway noaster noparen
parmest, saving("$outreg/est_entry_age_wage.dta",replace)

preserve
	use $outreg/est_entry_age_n4, clear
	rename (estimate stderr min95 max95) (estimate_a stderr_a min95_a max95_a)
	merge 1:1 parm using $outreg/est_entry_age_wage_n4
	rename (estimate stderr min95 max95) (estimate_b stderr_b min95_b max95_b)

	replace parm = "21.age" if _n==46
	foreach var in estimate_a estimate_b stderr_a stderr_b min95_a max95_a min95_b max95_b {
		replace `var'=0 if parm=="21.age"
	}

	sort parm
	gen age = _n + 19

	graph twoway rcap min95_a max95_a age, lpattern(dash) lwidth(thin) lcolor(gray) ///
	||	rcap min95_b max95_b age, lpattern(solid) lwidth(thin) lcolor(black) ///
	||	scatter estimate_a age, mc(gray) msize(tiny) ///
	||	scatter estimate_b age, mc(black) msize(tiny) ///
	legend(order(3 "Raw Estimates" 4 "Estimates after controlling for prior wages")) graphregion(color(white)) xtitle("Age of Individual") ///
	saving($outreg/plot_entry_age_wage.gph, replace)
restore


*==========================================================================
* Figure A6: Wage Distributions of Non-Founders, Founders, and Successful Founders
*	.. 5-percentile bins
*	.. plot only founder - non-founders are constant 5% baseline
*==========================================================================

use "/projects9/dms-p1-988-hc/project45/output/prior_earn_w2", clear /* W2 workers (1% random sample) and founders */
gen pearn_real =.
forval i = 5(5)95 {
qui egen p`i' = pctile(earn_ann_prior_real), p(`i')
	qui replace pearn_real = `i' if earn_ann_prior_real >= p`i' & earn_ann_prior_real ~=.
	qui drop p`i'
}
replace pearn_real = 0 if earn_ann_prior_real ~=. & pearn_real ==.

preserve /* kernel density */
	bysort pearn_real: egen count_nonfounder = sum(nonfounder)
	bysort pearn_real: egen count_founder = sum(founder)
	bysort pearn_real: egen count_top1_abs3 = sum(top1_abs3)
	gen perc_nonfounder = count_nonfounder / 12410187
	gen perc_founder = count_founder / 2432616
	gen perc_top1_abs3 = count_top1_abs3 / 20370
	bysort pearn_real: keep if _n == 1
		
	twoway (scatter perc_founder pearn_real) ///
		(scatter perc_top1_abs3 pearn_real), ///
		saving($outreg/density_wage_founders.gph, replace)

restore


*==========================================================================
* Table A1: Founder Age - Perceptions from Media & Two Prominent VCs
*      This table uses publicly available data.
*===========================================================================

** TechCrunch awards (public data)
clear
use TechCrunch
sum age, det 

** Inc and Entrepreneurs magazines (public data)
clear
use Magazines
sum age, det

** Sequoia (public data)
clear
use Sequoia
sum age, det 

** Matrix (public data)
clear
use Matrix
sum age, det


*==========================================================================
* Table A3: Founder Age - Oldest and Youngest Tech Sectors
*	 ..5 youngest and oldest in high-tech
*==========================================================================

qui setup
keep if high_tech_all == 1
bysort naics4: egen age_naics4 = mean(age)
bysort naics4: egen n_naics4 = sum(1)
bysort naics4: keep if _n == 1
sort age_naics4
list naics4 age_naics4



*==========================================================================
* Table A4: Mean Founder Age by Calendear Year of Firm's Founding
*	 ..2007-2014 ACROSOS
*	.. All, High-Tech, VC, PAtent, Top1%, Exit
*==========================================================================

qui setup

gen group1 = 1 /*All*/
gen group2 = (high_tech_all==1) /*High-Tech*/
gen group3 = (vc_ever==1) /*VC backed*/
gen group4 = (pat_ever==1) /*Patenting firms*/
gen group5 = (top1_abs3==1) /* Top 1% employment in 3 years*/
gen group6 = (exit==1) /* Exit via acquisition or IPO */

foreach i of numlist 2007/2014 {
	summ age if year==`i' & group1==1
	summ age if year==`i' & group2==1
	summ age if year==`i' & group3==1
	summ age if year==`i' & group4==1
	summ age if year==`i' & group5==1
	summ age if year==`i' & group6==1
}



*==========================================================================
* Table A5: Minimum and Maximum Ages within Founder Teams
*	.. Panel A: K-1 Owner-Workers
*	.. Panel B: Top 3 and Lowest earners (for both K-1 and C-corp firms)
*=========================================================

** Panel A: Firms based on K-1 Owner Workers
use "/projects9/dms-p1-988-hc/project45/output/k1_founders_2007_2014", clear
gen exit = (acq == 1 | ipo == 1)

* Min and Max age
bysort ein year: egen age_min = min(age)
bysort ein year: egen age_max = max(age)

gen all = 1
bysort ein year: keep if _n ==1

foreach var in all high_tech_all vc_ever pat_ever top1_abs5 topx1_abs5 exit {
display "=============== `var' =================="
	preserve
	keep if `var' == 1
	summ age_min
	summ age_max
	restore
}


** Panel B: Firms based on Top 3 first-year earners (ie initial founding team) from both K-1 and C-corps
use "/projects9/dms-p1-988-hc/project45/output/k1_w2_2007_2014", clear
append using "/projects9/dms-p1-988-hc/project45/output/ccorp_w2_2007_2014"
gen exit = (acq == 1 | ipo == 1)

* Min and Max age
bysort ein year: egen age_min = min(age)
bysort ein year: egen age_max = max(age)

gen all = 1
bysort ein year: keep if _n ==1

foreach var in all high_tech_all vc_ever pat_ever top1_abs5 topx1_abs5 exit {
display "=============== `var' =================="
	preserve
	keep if `var' == 1
	summ age_min
	summ age_max
	restore
}
