

library("corpcor")
library("Matrix")
library("gdata")

###############################################################
############################################################### 
# DEFINE INPUTS OF THE FUNCTION 
############################################################### 

# y                 Data 
# group             Vector that indicates group membership
# nM                Number of mixtures 
# nF                Number of factors (including all periods) 
# nI                Number of instruments 
# freelambda        Matrix indicating normalized and free factor loadings (1=normalized, 2=free)
# mstep.start       Starting values for EM algorithm (M-step in particular) 
# param.start       Starting values for minimum distance estimator 
# conv              Convergence criterion for EM algorithm 


###############################################################
# DEFINITION OF FUNCTION STARTS HERE
############################################################### 
estim.meas.model.nonparametric.group        <- function(y, group, nM, nF, nI, conv, freelambda, freeuniq, freemean, mstep.start, param.start){

  
###############################################################
###############################################################
# DEFINE PARAMETERS THAT WILL BE USED THROUGHOUT 
# nZ                Number of measurements
nZI <- ncol(y)       
scanNA <- function(x) c(which(complete.cases(x)==TRUE))
  
  
  
##############################################################
# 2.1. Define objects to fill in the "param" vector properly 
##############################################################
# Param is the vector of parameter solution of the min dist estimator 
# It contains the following (in this order)
# Variance and covariance of the measurement error terms 
# 1/2 * nF*(nF+1)*nM elements of the variance-covariance matrixes of the mixtures 
# nF*(nM-1) elements of the vectors of means for the nM-1 first mixtures 
# Factor loadings of the measurement system 
  endEps     <- length(which(freeuniq[upper.tri(freeuniq, diag=T)]==1))
  upcorr     <- which(freeuniq[upper.tri(freeuniq, diag=T)]==1)
  
  startCov1   <- endEps + 1
  endCov1     <- endEps + 0.5*nF*(nF+1)
  startCov2   <- endCov1 + 1
  endCov2     <- endCov1 + 0.5*nF*(nF+1)
  startCov     <- c(startCov1, startCov2)
  endCov     <- c(endCov1, endCov2)
  #covpos     <- matrix(seq(startCov, endCov, (0.5*nF*(nF+1))), nM, nG)
  
  startMean  <- endCov2 + 1
  endMean    <- startMean - 1 + nF
  
  nLambda    <- length(which(freelambda==2))
  
  startLambda <- endMean + 1 
  endLambda   <- startLambda - 1 + nLambda
  lambdalong  <- rep(0, nZ*nF)
  lambdalong[which(freelambda==1)]  <- 1 
  
  
  
  
##############################################################
# 2.2. Define function to optimize over 
##############################################################
mindistance      <-function(param) {
    
# 2.2.1 Fill in various objects with elements of param  
##############################################################
 # Variances of the uniquenesses 
eps_par                    <- matrix(0, nZ, nZ)
eps_par[upper.tri(eps_par, diag=T)][upcorr] <- param[1:endEps]
eps_par                    <- eps_par + t(eps_par) - diag(nZ) * diag(eps_par)
    
# Variance-covariance matrices of the nM mixtures 
Lcovfactor_par    <- list()
covfactor_par     <- list()
    
for (g in 1:nG){
  Lcovfactor_par[[g]] <- list()  
  covfactor_par[[g]]  <- list()  
  Lcovfactor_par[[g]]                              <- matrix(0, nF, nF)
  lowerTriangle(Lcovfactor_par[[g]], diag=TRUE)    <- param[startCov[g]:endCov[g]]
  covfactor_par[[g]]                               <- Lcovfactor_par[[g]] %*% t(Lcovfactor_par[[g]])

} 
    
# Means of the nM-1 mixtures  
# For one group we assume the overall mean to be 0, but not for the other groups 
meanfactor_par          <- param[startMean:endMean]
   
    
# Factor loadings (same across groups)
lambdalong[which(freelambda==2)]  <- param[startLambda:endLambda]
lambda_par                        <- matrix(lambdalong, nZ, nF)
    
    
# 2.2.2 Extract output of the EM alrogithm and compute estimated 
# means and variance-covariances 
############################################################## 
# Moments in the data 
    mean_hat      <- list()
    cov_hat       <- list()
    prop_hat      <- list()
    
    for (g in 1:nG){
     cov_hat[[g]]       <- cov(y[group==g,], use = "pairwise.complete.obs")
     mean_hat[[g]]      <- colMeans(y[group==g,], na.rm=TRUE)
    } 
    
    
    
    
# Variance-covariance matrix 
    cov_par  <- list()
    for (g in 1:nG){
    cov_par[[g]] <- list()
    cov_par[[g]] <-lambda_par%*%covfactor_par[[g]]%*%t(lambda_par) + eps_par
    } 

    cov <- 0
    for (g in 1:nG){
      cov     <- cov + sum(sum((cov_hat[[g]]-cov_par[[g]])^2))
    }

    
    
     
# Difference in means between treatment and controls
mean_par <- list()
mean <- sum(((mean_hat[[2]] - mean_hat[[1]]) - lambda_par %*% meanfactor_par)^2)

# Sum of least squares 
sos <- cov + mean
    
return(sos)
}
  
  
  
##############################################################
# 3. Perform minimum distance estimator  
##############################################################
out <- optim(param.start,mindistance, method=c("L-BFGS-B"), control=list(maxit=10000),
               lower=c(rep(0, nZ), rep(-Inf, length(param.start)-nZ)), upper=rep(Inf, length(param.start)))
  
##############################################################
# 4. Organize output from the minimum distance estimator
##############################################################
  # Info to organise output 
startCov1   <- endEps + 1
endCov1     <- endEps + 0.5*nF*(nF+1)
startCov2   <- endCov1 + 1
endCov2     <- endCov1 + 0.5*nF*(nF+1)
startCov     <- c(startCov1, startCov2)
endCov     <- c(endCov1, endCov2)

startMean  <- endCov2 + 1
endMean    <- startMean - 1 + nF

nLambda    <- length(which(freelambda==2))

startLambda <- endMean + 1 
endLambda   <- startLambda - 1 + nLambda
lambdalong  <- rep(0, nZ*nF)
lambdalong[which(freelambda==1)]  <- 1 



# Variance of measurement error 
eps_est  <- out$par[1:endEps]

# Variance covariance matrix 
Lcovfactor_est    <- list()
covfactor_est     <- list()

for (g in 1:nG){
  Lcovfactor_est[[g]]                         <- matrix(0, nF, nF)
  lowerTriangle(Lcovfactor_est[[g]], diag=TRUE) <- out$par[startCov[g]:endCov[g]]
  covfactor_est[[g]]                          <- Lcovfactor_est[[g]] %*% t(Lcovfactor_est[[g]])
} 

# Means 
mean_est              <- out$par[startMean:endMean]

# Factor Loadings 
lambdalong[which(freelambda==2)]  <- out$par[startLambda:endLambda]
lambda_est  <- matrix(lambdalong, nZ, nF)

##############################################################
# 6. Group output to save from the function  
##############################################################
est <- list(eps=eps_est, cov=covfactor_est, mean=mean_est, lambda=lambda_est)
all <- list(out, est)

}