###################################################################
# Name              : MASTER.R
# 
# Description       : Master program running: 
#                     1. the estimation of the measurement system, 
#                     2. the estimation of the factor scores and 
#                     3. the estimation of the investment and production functions  
###################################################################



###################################################################
# SETTINGS 
###################################################################

rm(list=ls())                  # Clear all objects from memory
set.seed(08072018)             # Random number generator seed (This is the seed number that needs to be used to reproduce the work on R Version 3.3.1)

# Load packages 
library("MASS")
library("mixtools")
library("mvtnorm")
library("minpack.lm")
library("corpcor")
library("gdata")
library("Matrix")
library("ks")
library("psych")

###################################################################
# PROGRAMME OPTIONS
###################################################################

# Boostrap settings 
Bootstrap          <- 1                                      # 1 if run the bootstrap; 0 otherwise
onlyboot           <- 0                                      # 1 if we only want to perform the bootstrap; 0 if we want to estimate the model on true data and bootstrap data
node               <- 1                                      # Number of nodes/R sessions on which to run the estimation in parallel for bootstrapped samples
bsample            <- 1000                                 # Number of bootstrap samples on each node
pos                <- seq(1,1,1)                             # Nodes used for the boostrap
nameBootFM         <- paste("BootFM", node, ".R", sep="")    # Name of the output from estimation of the measurement system on the n-th node
nameBootFS         <- paste("BootFS", node, ".R", sep="")    # Name of the output from estimation of the factor scores on the n-th node

# EM estimation algorithm settings
conv               <- 1e-6                                   # Convergence criterion

# Options for robustness checks (main results are with all three options set to 0)
testerFE           <- 0                                     # Tester FE removed from raw data 
puresample         <- 0                                     # Sample with pure control (no nutrition) and pure treatment (only stimulation) groups
invar              <- 0                                    # Allowing for some measurement variance between control and treated groups  


###################################################################
# DIRECTORIES
###################################################################

# Main working directory 
dir                 <- c("AER-2015-0183_data_appendix/")   # Name of your main working directory 

# Sub-directories 
dir_anal           <- paste(dir,c("code/R"), sep="") 
dir_functions      <- paste(dir,c("code/R/functions"), sep="")
dir_pf             <- paste(dir,c("code/R/specsPF"), sep="")
dir_data           <- paste(dir,c("data"), sep="")


if (testerFE == 1){
  dir_data       <- paste(dir_data, "_testerFE", sep="")    
}
if (puresample ==1){
  dir_data       <- paste(dir_data, "_puresample", sep="")
}
if (invar ==1){
  dir_data       <- paste(dir_data, "_invar", sep="")
}


###################################################################
# CALL FUNCTIONS
###################################################################

setwd(dir_functions)
source("call.data_func.R")
source("draw.factor.noins_func.R")
source("estim.meas.model.group_func.R")
source("estim.meas.model.group.intercept.invariance_func.R")
source("sig2noise.func.R")
source("bartlett.func.R")
source("bartlett.correction.mixt.func.R")
source("bartlett.correction.mixt.interacttreat.func.R")
source("pf.ols.func.R")
source("pf.ols.fullyinteract.func.R")
source("first.stage.func.R")
source("first.stage.interacttreat.func.R")
source("rdform.func.R")
source("rdform.interacttreat.func.R")
source("md.flexfunc.R")
source("md.1inv.flexfunc.R")
source("md.fullyinteract.1inv.flexfunc.R") 

###################################################################
# MEASUREMENT SYSTEM SPECIFICATION 
###################################################################

nameFM             <- c("latentfactors_noinstruments")                    # Name of the measurement system specification file
if (invar ==1){
  nameFM             <- c("latentfactors_noinstruments_invar")                    # Name of the measurement system specification file
}

inputFM            <- paste("specsFM/",nameFM, ".R", sep="")              # Location of the file with all the details of the specification

# Define directories where we will save the results for that measurement system specification 
dir_outputFM       <- paste(dir,c("output/"), nameFM, sep="")   

if (testerFE == 1){
dir_outputFM       <- paste(dir,c("output/"), nameFM, "/testerFE", sep="")
}

if (puresample ==1){
  dir_outputFM    <- paste(dir,c("output/"), nameFM, "/puresample", sep="")
}


###################################################################
# LOAD DATA AND SPECIFICATION OF THE MEAUSUREMENT SYSTEM
##################################################################

# Load data
setwd(dir_data)                                                         # Open directory that contains the data (true data and bootstrap datasets)
measures <- call.data("measures.csv")

# Load details of the empirical specification
setwd(dir_anal)                                                         # Open directory that contains the detail of the specification
source(inputFM)                                                         # Load  file specifying the measurement system

################################################################### 
# ESTIMATE MEASUREMENT SYSTEM 
################################################################### 

# Run estimation program 
setwd(dir_anal)
source("EstimateMeasurementSystem.R")


# Produce tables with estimates of the measurement system 
setwd(dir_anal)
if (invar==0){
  source("AssembleBootstrapFM.R")
}

if (invar==1){
   source("AssembleBootstrapFM_invar.R")
}



# Compute signal to noise ratios for all the measures 
setwd(dir_anal)
source("SignalToNoiseRatio.R")


# Graph kernel densities of the estimated latent factors
setwd(dir_anal)
source("KernelDensities.R")
 

###################################################################
# ESTIMATE FACTOR SCORES 
###################################################################

setwd(dir_anal)
source("EstimateFactorScores.R")


# Assemble them 
setwd(dir_anal)
source("AssembleBootstrapFS.R")

################################################################### 
# OLS estimates of the production function 
################################################################### 
# Main specification of the production function
namePF            <- c("CobbDouglas_OLS")
dir_outputPF      <- paste(dir,c("output/"), nameFM, "/", namePF, sep="")
setwd(dir_pf)
source(paste(namePF, ".R", sep=""))


################################################################### 
# First stage estimates and IV estimates of the production function 
# using toyprice, foodprice and conflict 
# as instruments 
################################################################### 
# First stage 
nameFirstStage <- c("FirstStage_toyprice_foodprice_conflict")
dir_outputPF      <- paste(dir,c("output/"), nameFM,  "/",  nameFirstStage, sep="")
setwd(dir_pf)
source(paste(nameFirstStage, ".R", sep=""))

# IV estaimtes of Cobb Douglas PF with both investments 
setwd(dir_pf)
posExo       <- c(1:7)
posInv       <- c(8:10)
source("CobbDouglas_IV.R")


# IV estimates of Cobb Douglas PF with both investments and excluding 
# treatment from the production function  
setwd(dir_pf)
posExo       <- c(1,3:7)
posInv       <- c(8:10,2)
source("CobbDouglas_IV_notreatment.R")


# IV estimates of Cobb Douglas PF with one investment (material for cognitive skill; 
# time for socio-emotional skill)
setwd(dir_pf)
posExo       <- c(1:7)
posInv       <- c(8:10)
source("CobbDouglas_IV_parsinv.R")


# IV estimates of Cobb Douglas PF with one investment (material for cognitive skill; 
# time for socio-emotional skill) and excluding treatment from the 
# production function  
setwd(dir_pf)
posExo       <- c(1,3:7)
posInv       <- c(8:10,2)
source("CobbDouglas_IV_notreatment_parsinv.R")


################################################################### 
# First stage estimates and IV estimates of the production function 
# using toyprice and foodprice 
# as instruments 
################################################################### 
# First stage 
nameFirstStage <- c("FirstStage_toyprice_foodprice")
dir_outputPF      <- paste(dir,c("output/"), nameFM,  "/",  nameFirstStage, sep="")
setwd(dir_pf)
source(paste(nameFirstStage, ".R", sep=""))


# IV estimates of Cobb Douglas PF with one investment (material for cognitive skill; 
# time for socio-emotional skill)
setwd(dir_pf)
posExo       <- c(1:7)
posInv       <- c(8:9)
source("CobbDouglas_IV_parsinv.R")


# IV estimates of Cobb Douglas PF with one investment (material for cognitive skill; 
# time for socio-emotional skill) and excluding treatment from the 
# production function  
setwd(dir_pf)
posExo       <- c(1,3:7)
posInv       <- c(8:9,2)
source("CobbDouglas_IV_notreatment_parsinv.R")


################################################################### 
# Estimates of first stage using toyprice, foodprice and conflict 
# as instruments and where all variables are interacted with 
# treatment dummy (Appendix Table E.1)
################################################################### 
nameFirstStage <- c("FirstStageFullyInteracted_toyprice_foodprice_conflict")
dir_outputPF      <- paste(dir,c("output/"), nameFM,  "/",  nameFirstStage, sep="")
setwd(dir_pf)
source(paste(nameFirstStage, ".R", sep=""))


################################################################### 
# Estimates of model with interactions between all variables of the PF 
# and treatment dummy (Appendix Table E.2)
################################################################### 
# Production function where all parameters are interacted with treatment 
namePF            <- c("CobbDouglas_OLS_fullyinteracted")
dir_outputPF      <- paste(dir,c("output/"), nameFM, "/", namePF, sep="")
setwd(dir_pf)
source(paste(namePF, ".R", sep=""))

# First stage and reduced form corresponding to PF with interactions 
# only using toyprice and food price as instruments 
nameFirstStage <- c("FirstStageForIVInteracted_toyprice_foodprice")
dir_outputPF      <- paste(dir,c("output/"), nameFM,  "/",  nameFirstStage, sep="")
setwd(dir_pf)
source(paste(nameFirstStage, ".R", sep=""))


# IV estimates of Cobb Douglas PF with one investment (material for cognitive skill; 
# time for socio-emotional skill) where all parameters are interacted with 
# treatment 
setwd(dir_pf)
posExoFS    <- c(1:7)
posInvFS    <- c(8:9)
posExoRF    <- c(1:12)
posInvRF    <- c(13:16)
source("CobbDouglas_IV_parsinv_fullyinteracted.R")



################################################################### 
# Estimates of the model with strata dummies 
################################################################### 
# OLS estimates of the production function 
namePF            <- c("CobbDouglas_OLS_strata")
dir_outputPF      <- paste(dir,c("output/"), nameFM, "/", namePF, sep="")
setwd(dir_pf)
source(paste(namePF, ".R", sep=""))

# First stage with toyprice, foodprice, and conflict as instruments 
nameFirstStage <- c("FirstStage_toyprice_foodprice_conflict_strata")
dir_outputPF      <- paste(dir,c("output/"), nameFM,  "/",  nameFirstStage, sep="")
setwd(dir_pf)
source(paste(nameFirstStage, ".R", sep=""))

# IV estimates of the production function
setwd(dir_pf)
posExo       <- c(1:9)
posInv       <- c(10:12)
source("CobbDouglas_IV.R")



