function [dPhi,exfl,eigs] = NKRF_SlvMod_Full(prsall,OP)
%
% This function obtains the perturbation solution given the reduced-form
% parameters in prsall.
% 
% Inputs:
%   prsall      reduced-form parameters
%   OP          struct variable containing non-model variables/parameters
%
% Outputs:
%   dPhi        vector of derivatives of policy function at the solution
%   exfl        flag = -1 if solution is indeterminate
%   eigs        matrix where each row corresponds to an eigenvalue of the
%                   linearized system, with real part in the first column,
%                   imaginary part in the second, and modulus in the third
%
% For Beaudry, Galizia, and Portier, Putting the Cycle Back into Business
% Cycle Analysis (2019).

%%  Initialize

% Note: the order of the reduced-form parameters in prs is:
%       del psi xi al2 al1 rho   Dth1       Dth2 Dth3
%      | "linear" parameters  |   =0  | "non-linear" params

npr = 6;                % number of "linear" parameters

% extract "linear" parameters
prs = prsall(1:npr);    
del = prs(1);
psi = prs(2);
xi = prs(3);
al2 = prs(4);
al1 = prs(5);
rho = prs(6);

% extract "non-linear" parameters
Dth = prsall(npr+1:end)';

exfl = 0;       % initialize exit flag to 0

%% Solve to first order

% transition matrix for linearized system
A = [1-del, 0, psi;   ...
    0, 0, 1; ...
    xi*al1, -xi*al2, xi;  ...
    ];

% eigendecomposition of A
[V,Lam] = eig(A);
Lam = diag(Lam);
eigs = [real(Lam),imag(Lam),abs(Lam)];

% real & >1 eigenvalues:
iUSeig = (real(Lam)>1) & (imag(Lam)==0);    % indices
USeig = Lam(iUSeig);                        % values
nUSeig = nnz(iUSeig);                       % number of them
if nUSeig > 1       % if more than 1
    iUSeig = find(abs(Lam)==max(abs(USeig)),1); % index of the largest in modulus
elseif nUSeig == 0  % elseif there are none
	exfl = -1;          % flag indicating indeterminacy
    
    % compute a penalty to add to the objective function and return it in
    % place of dPhi, then exit
    mxrlLam = max(Lam(imag(Lam)==0));
    dPhi = 1e6*(2-mxrlLam);
    return;
end

iV = inv(V);        % inverse of eigenvector matrix

% first-order approximation coefficients:
Ac = real(-iV(iUSeig,3)\iV(iUSeig,1:2));    % on endogenous state variables
Bc = -xi/(rho-Lam(iUSeig));                 % on exogenous state variable

dPhi = zeros(OP.ntrmslv,1);         % allocate memory

% load first-order coefficients into dPhi
dPhi(1) = Ac(1);            
dPhi(2) = Ac(2);
dPhi(3) = Bc;

%% 2nd- and 3rd-order derivatives of solution

mnord = sum(OP.expnsslv,2);     % total order of each monomial
ctr = find(mnord==2,1);         % index of the first non-linear monomial

for j = 2:OP.ord        % for each non-linear order
    
    % call function to get matrices characterizing solution for j-th-order
    % derivatives
    [B1,B2] = feval(['DervFll_ord' num2str(j) '_fn'],dPhi(1:ctr-1),Dth,prs);
    
    if rcond(B1) < 1e-10    % if B1 is singular
        dPhik = pinv(B1)*B2;    % use Moore-Penrose inverse to solve system
    else                    % otherwise
        dPhik = B1\B2;          % solve linear system
    end
    
    nk = numel(dPhik);          % number of monomials for j-th order
    dPhi(ctr:ctr+nk-1) = dPhik; % load j-th-order derivatives into dPhi
    ctr = ctr+nk;               % advance counter to index corresponding to 
                                    % first (j+1)-th-order derivative
end


