## ======================================
## Figure 9: Foodborne hospitalizations per 100,000 people comparing J&L's disease selection with and without
## campylobacter in LA, Southern CA, and CA excluding LA (CA)
## ======================================

  rm(list=ls())
  
  #Assume base R is running
  my.wd <- getSrcDirectory(function(x){x})
  
  #Check for errors
  if(grepl("error", tolower(class(my.wd)[1])) | my.wd==""){
    #Try to access working directory through R Studio API
    my.wd <- tryCatch(dirname(rstudioapi::getActiveDocumentContext()$path),
                      error = function(e) e)
    
  }
  
  #Set working directory
  setwd(my.wd)
  
## ======================================
## Read in data
## ======================================
  
  #Hospitalizations
  hosp.cty.yr.1983.2009 <- read.csv("../Data/hosp_county_year_1983_2009.csv", stringsAsFactors = FALSE)
  
  #Population
  county.pop.1964.2015 <- read.csv("../Data/county_population_1964_2015.csv", stringsAsFactors = FALSE)
  
  #Keep only 1993-2009 because campylobacter gets its own ICD code
  hosp.cty.yr.1983.2009 <- subset(hosp.cty.yr.1983.2009, year>=1993)
  
## ==========================
## Aggregate yearly population
## ==========================  

  ## So Cal county names
  socal <- c("Imperial", "Kern","Orange","San Bernardino","San Diego", "San Luis Obispo","Santa Barbara", "Riverside", "Ventura")

  #Total yearly population in CA non-la
  ca0.pop <- aggregate(pop~year, FUN = sum,
                       data=subset(county.pop.1964.2015, !Name == "Los Angeles"))
  
  #Total yearly population in LA
  la.pop <- aggregate(pop~year+ifelse(Name=="Los Angeles", 1, 0), data = county.pop.1964.2015, FUN = sum)
  names(la.pop)[2] <- "la"
  
  #Total yearly population in SoCal
  socal.pop <- aggregate(pop~year+ifelse(Name%in%socal, 1, 0), data = county.pop.1964.2015, FUN = sum)
  names(socal.pop)[2] <- "socal"
  
## ==========================
## Aggregate illness counts 
## ==========================  

  ## Add county indicators
  hosp.cty.yr.1983.2009$la <- as.numeric(hosp.cty.yr.1983.2009$county.name=="Los Angeles")
  hosp.cty.yr.1983.2009$socal <- as.numeric(hosp.cty.yr.1983.2009$county.name%in%socal)
  
  ## Add camp to disease selection
  hosp.cty.yr.1983.2009$jl.codes.camp <- hosp.cty.yr.1983.2009$jl.codes.mdc6 + hosp.cty.yr.1983.2009$camp
  
  #Names of illnesses
  fbi.names <- c("jl.codes.mdc6","jl.codes.camp")

  #Aggregate counts
  year.ca0.agg <- aggregate(formula(paste0("cbind(", paste0(fbi.names, collapse=","),")~year")), 
                            data = subset(hosp.cty.yr.1983.2009, la == 0), 
                            FUN = sum)
  year.la.agg <- aggregate(formula(paste0("cbind(", paste0(fbi.names, collapse=","),")~year+la")), 
                           data = subset(hosp.cty.yr.1983.2009, la == 1), 
                           FUN = sum)
  year.socal.agg <- aggregate(formula(paste0("cbind(", paste0(fbi.names, collapse=","),")~year+socal")), 
                              data = subset(hosp.cty.yr.1983.2009, socal == 1), 
                              FUN = sum)
  
  #Add time varying population
  year.la.agg <- merge(year.la.agg, la.pop, by = c("la", "year"))
  year.socal.agg <- merge(year.socal.agg, socal.pop, by = c("socal", "year"))
  year.ca0.agg <- merge(year.ca0.agg, ca0.pop, by = c("year"))
  
  #Create rates
  for(fbi in fbi.names){
    year.la.agg[,paste0("rate.", fbi)] <- 100000*year.la.agg[,c(fbi)]/year.la.agg$pop
    year.socal.agg[,paste0("rate.", fbi)] <- 100000*year.socal.agg[,c(fbi)]/year.socal.agg$pop
    year.ca0.agg[,paste0("rate.", fbi)] <- 100000*year.ca0.agg[,c(fbi)]/year.ca0.agg$pop
  }

## ==========================
## Plot 
## ==========================  

  par(mfrow=c(1,2), mar=c(3,3,2,1), mgp=c(1.5,0.5,0), tcl=-0.3)
  
  #Colors
  la.color <- rgb(1,0,0,0.8)
  ca.color <- rgb(0,0,1,0.8)
  
  ## Original disease selection
  plot(year.la.agg$year, year.la.agg$rate.jl.codes.mdc6, type = "n", col = ca.color,
       ylab = "Rate", xlab = "Year", main = "J&L Disease Selection", ylim = c(2,6))
  rect(xleft = 1995, xright = 1999, ytop = 8, ybottom = 0, border = NA, col = rgb(0,0,0,0.1))
  abline(v=1997.5,col="white")
  lines(year.ca0.agg$year, year.ca0.agg$rate.jl.codes.mdc6, col = ca.color, lty=5,lwd=1)
  lines(year.la.agg$year, year.la.agg$rate.jl.codes.mdc6, col = la.color, lwd=1.5)
  lines(year.socal.agg$year, year.socal.agg$rate.jl.codes.mdc6, col = rgb(1,0,1,1), lty=5,lwd=1)
  text(x = 1994, y = 3.1, label = "CA", col = ca.color, cex=0.85)
  text(x = 1994, y = 4.5, label = "LA", col = la.color, cex=0.85)
  text(x = 1995, y = 3.7, label = "Southern CA", col = "purple", cex=0.85)
  text(1997,5.7,"J&L\nwindow\n1995-99",col=rgb(0,0,0,0.75),cex=0.7)
  
  
  #Plus Campylobacter
  plot(year.la.agg$year, year.la.agg$rate.jl.codes.camp, type = "n", col = ca.color,
       ylab = "Rate", xlab = "Year", main = "J&L Plus Campylobacter", ylim = c(2.6,7))
  rect(xleft = 1995, xright = 1999, ytop = 8, ybottom = 0, border = NA, col = rgb(0,0,0,0.1))
  abline(v=1997.5,col="white")
  lines(year.la.agg$year, year.la.agg$rate.jl.codes.camp, col = la.color, lwd=1.5)
  lines(year.ca0.agg$year, year.ca0.agg$rate.jl.codes.camp, lty=5, col=ca.color,lwd=1)
  lines(year.socal.agg$year, year.socal.agg$rate.jl.codes.camp, col = rgb(1,0,1,1), lty=5,lwd=1)


