## ======================================
## Appendix A Table 7: Dropped counts in Simon et al (2005)
## ======================================

  rm(list=ls())
  
  #Assume base R is running
  my.wd <- getSrcDirectory(function(x){x})
  
  #Check for errors
  if(grepl("error", tolower(class(my.wd)[1])) | my.wd==""){
    #Try to access working directory through R Studio API
    my.wd <- tryCatch(dirname(rstudioapi::getActiveDocumentContext()$path),
                      error = function(e) e)
    
  }
  
  #Set working directory
  setwd(my.wd)

## ======================================
## Load packages and functions
## ======================================

  #Read in function to reshape data for modeling
  source("Functions/prepModelData.R")
  source("Functions/installPackageNotFound.R")
  source("Functions/formatSig.R")
  
  #Libraries for modeling
  installPackageNotFound("data.table")
  installPackageNotFound("plm")
  installPackageNotFound("stargazer")
  installPackageNotFound("lmtest")
  
## ======================================
## Read in data
## ======================================

  #Hospitalizations
  hosp.zip3.qtr.1983.2009 <- read.csv("../Data/hosp_zip3_quarter_1983_2009.csv", stringsAsFactors = FALSE)

## ==========================
## Reshape data for modeling 
## ==========================  

  ## Simon disease selection
  dta.simon.did <- prepModelData(dta = hosp.zip3.qtr.1983.2009, 
                                 fbi.name = "simon.codes", 
                                 control.name = "append", 
                                 years = 1993:2000,
                                 control.grp = FALSE,
                                 filter = FALSE)

## ==========================
## LA DiD with all observations
## ==========================  

  #LA treatment 
  dta.simon.did$prop.treated <- dta.simon.did$prop.la
  
  #1) DiD LA
  did.la <- plm(logCount ~ I(as.numeric(year>=1998)*prop.treated) , 
                index = c("geo", "year.qtr"), 
                model="within", effect="twoways", 
                data = dta.simon.did)
  
  did.la.se <- coeftest(did.la, vcov=vcovHC(did.la, type="HC0", cluster="group"))
  did.la.r2 <- formatSig(summary(lm(logCount ~ I(as.numeric(year>=1998)*prop.treated) +
                               factor(year.qtr) + factor(geo), data = dta.simon.did))$r.squared, 2)
  
## ==========================
## LA DiD dropping quarters with fewer than 3 obs
## ==========================  

  did.la.dropped <- plm(logCount ~ I(as.numeric(year>=1998)*prop.treated) ,
                index = c("geo", "year.qtr"),
                model="within", effect="twoways",
                data = subset(dta.simon.did, simon.codes >=3 ))
  
  did.la.dropped.se <- coeftest(did.la.dropped, vcov=vcovHC(did.la.dropped, type="HC0", cluster="group"))
  did.la.dropped.r2 <- formatSig(summary(lm(logCount ~ I(as.numeric(year>=1998)*prop.treated) +
                                       factor(year.qtr) + factor(geo), data = subset(dta.simon.did, simon.codes >=3 )))$r.squared, 2)
## ==========================
## Latex output for LA vs LA Dropped
## ==========================  

  #Labels
  cov.label <- c("Grade Cards")
  title <- c("Table 7: Comparison of DID model with LA as the treated county using all observations included and dropping quarters with fewer than 3 hospitalization counts.")

  #Print table
  stargazer(did.la,
            did.la.dropped,
            coef = list(round(did.la$coefficients, 2), 
                        round(did.la.dropped$coefficients, 2)),
            se = list(did.la.se[,2], 
                      did.la.dropped.se[,2]),
            p = list(did.la.se[,4], 
                     did.la.dropped.se[,4]),
            covariate.labels = cov.label,
            title = title,
            digits = 2,
            dep.var.labels = "",
            dep.var.caption = "",
            notes.append = FALSE,
            column.sep.width = "1pt",
            no.space = TRUE,
            omit.stat = c("rsq", "adj.rsq", "f"),
            notes.align = "l",
            type = "text",
            add.lines = list(c("R2", did.la.r2, did.la.dropped.r2)))
