## ======================================
## Table 4: Marginal effects of restaurant grading in LA with Southern CA as the control group.
## ======================================

  rm(list=ls())
  
  #Assume base R is running
  my.wd <- getSrcDirectory(function(x){x})
  
  #Check for errors
  if(grepl("error", tolower(class(my.wd)[1])) | my.wd==""){
    #Try to access working directory through R Studio API
    my.wd <- tryCatch(dirname(rstudioapi::getActiveDocumentContext()$path),
                      error = function(e) e)
    
  }
  
  #Set working directory
  setwd(my.wd)

## ======================================
## Load packages and functions
## ======================================

  #Read in function to reshape data for modeling
  source("Functions/prepModelData.R")
  source("Functions/installPackageNotFound.R")
  source("Functions/formatSig.R")
  source("Functions/calculateFRatio.R")
  
  #Libraries for modeling
  installPackageNotFound("data.table")
  installPackageNotFound("plm")
  installPackageNotFound("stargazer")
  installPackageNotFound("lmtest")

## ======================================
## Read in data
## ======================================

  #Hospitalizations
  hosp.zip3.qtr.1983.2009 <- read.csv("../Data/hosp_zip3_quarter_1983_2009.csv", stringsAsFactors = FALSE)

## ==========================
## Run models comparing LA to SoCal
## ==========================  

  #Observation window lists
  obs.window.list <- list(c(1995:1999), c(1993:2009))
  
  #Subset to LA and SoCal zip3s only
  this.model.data <- subset(hosp.zip3.qtr.1983.2009, prop.socal>0 | prop.la>0)
    
  #Loop through each observation window
  for(obs.window in obs.window.list){

    ## JL original disease selection 
    dta.agg.jl <- prepModelData(dta = this.model.data, 
                                fbi.name = "jl.codes.mdc6", 
                                control.name = "control.jl.codes", 
                                years = obs.window,
                                control.grp = TRUE,
                                filter =FALSE)
    
    #Name treatment variables (for latex table)
    dta.agg.jl$treatment.m <- dta.agg.jl$disease.type*dta.agg.jl$m
    dta.agg.jl$treatment.v <- dta.agg.jl$disease.type*dta.agg.jl$v
    
    ## ======================================
    ## Triple diff table
    ## ======================================
    
    #Number of observations (same for all models)
    n.obs <- nrow(dta.agg.jl)
    
    #Subset to foodborne 
    did.data <- subset(dta.agg.jl, disease.type == 1)
    
    #Rename variables to match triple diff data (for latex table)
    names(did.data)[which(names(did.data)=="m")] <- "treatment.m"
    names(did.data)[which(names(did.data)=="v")] <- "treatment.v"
    
    #Simple difference-in-differences model
    did.model <- plm(logCount ~ treatment.m + treatment.v , 
                     index = c("geo", "year.qtr"), 
                     model="within", effect="twoways", 
                     data= did.data)
    
    ## Adjust standard errors to cluster robust standard errors
    adjusted.se.did <- coeftest(did.model, vcov=vcovHC(did.model, type="HC0", cluster="group"))
    
    #Estimate model with fixed effects to adjust df
    did.lm <- lm(logCount ~ treatment.m + treatment.v +
                   factor(year.qtr) + factor(geo), data = did.data)
    
    #Calculate r2 with fixed effects
    r2.did <- formatSig(summary(did.lm)$r.squared, 2)
    
    #JL Spec
    jl.model <- plm(logCount ~ treatment.m + treatment.v + m + v , 
                    index = c("geo.type.pk", "year.qtr"), 
                    model="within", effect="twoways", data= dta.agg.jl)
    
    ## Adjust standard errors to cluster robust standard errors
    adjusted.se <- coeftest(jl.model, vcov=vcovHC(jl.model, type="HC0", cluster="group"))
    
    #Specify models with fixed effects to adjust degrees of freedom
    jl.lm <- lm(logCount ~ treatment.m + treatment.v + m + v  +
                  factor(year.qtr) + factor(geo.type.pk), data = dta.agg.jl)
    
    #Calculate r2 with fixed effects
    r2 <- formatSig(summary(jl.lm)$r.squared, 2)
    
    #Triple diff 
    jl.model.triple <- plm(logCount ~ treatment.m + treatment.v + I(as.numeric(year>=1998)*disease.type) + m + v , 
                           index = c("geo.type.pk", "year.qtr"), 
                           model="within", effect="twoways", data= dta.agg.jl)
    
    ## Adjust standard errors to cluster robust standard errors
    adjusted.se.triple <- coeftest(jl.model.triple, vcov=vcovHC(jl.model.triple, type="HC0", cluster="group"))
    
    #Specify models with fixed effects to adjust degrees of freedom
    triple.lm <- lm(logCount ~ treatment.m + treatment.v + I(as.numeric(year>=1998)*disease.type) + m + v +
                      I(as.numeric(year>=1998)*disease.type) +
                      factor(year.qtr) + factor(geo.type.pk), data = dta.agg.jl)
    
    #Calculate r2 with fixed effects
    r2.triple <- formatSig(summary(triple.lm)$r.squared, 2)
   
    
    #Save results
    assign(paste0("did.model.", min(obs.window), ".", max(obs.window), ".", "so.cal"), did.model)
    assign(paste0("adjusted.se.did.", min(obs.window), ".", max(obs.window), ".", "so.cal"), adjusted.se.did)
    assign(paste0("r2.did.", min(obs.window), ".", max(obs.window), ".", "so.cal"), r2.did)
    
    assign(paste0("jl.model.", min(obs.window), ".", max(obs.window), ".", "so.cal"), jl.model)
    assign(paste0("adjusted.se.", min(obs.window), ".", max(obs.window), ".", "so.cal"), adjusted.se)
    assign(paste0("r2.", min(obs.window), ".", max(obs.window), ".", "so.cal"), r2)
    
    assign(paste0("jl.model.triple.", min(obs.window), ".", max(obs.window), ".", "so.cal"), jl.model.triple)
    assign(paste0("adjusted.se.triple.", min(obs.window), ".", max(obs.window), ".", "so.cal"), adjusted.se.triple)
    assign(paste0("r2.triple.", min(obs.window), ".", max(obs.window), ".", "so.cal"), r2.triple)
  
    assign(paste0("n.obs.", min(obs.window), ".", max(obs.window), ".", "so.cal"), n.obs)
    
    assign(paste0("jl.lm.", min(obs.window), ".", max(obs.window), ".", "so.cal"), jl.lm)
    assign(paste0("triple.lm.", min(obs.window), ".", max(obs.window), ".", "so.cal"), triple.lm)
    
    
  }

## ======================================
## F test for model fit
## ======================================
  
  #Original time period s.cal
  f1 <- calculateFRatio(triple.lm.1995.1999.so.cal, jl.lm.1995.1999.so.cal)
  
  #Expanded time period s.cal
  f2 <- calculateFRatio(triple.lm.1993.2009.so.cal, jl.lm.1993.2009.so.cal)

## ======================================
## Latex output
## ======================================

  #Labels
  cov.label <- c("Foodborne  $\\times$ LA mandatory", "Foodborne $\\times$ LA voluntary", "Foodborne $\\times$ S.Cal.", "LA Mandatory disclosure", "LA Voluntary disclosure")
  title <- paste0("Table 4:  Marginal effects of restaurant grading in LA with Southern CA as the control group.")
  
  #Create latex output
  latex_output <- stargazer(did.model.1995.1999.so.cal,
                            jl.model.1995.1999.so.cal,
                            jl.model.triple.1995.1999.so.cal,
                            did.model.1993.2009.so.cal,
                            jl.model.1993.2009.so.cal,
                            jl.model.triple.1993.2009.so.cal,
                            coef = list(round(did.model.1995.1999.so.cal$coefficients, 2),
                                        round(jl.model.1995.1999.so.cal$coefficients, 2), 
                                        round(jl.model.triple.1995.1999.so.cal$coefficients, 2),
                                        round(did.model.1993.2009.so.cal$coefficients, 2),
                                        round(jl.model.1993.2009.so.cal$coefficients, 2), 
                                        round(jl.model.triple.1993.2009.so.cal$coefficients, 2)),
                            se = list(adjusted.se.did.1995.1999.so.cal[,2],
                                      adjusted.se.1995.1999.so.cal[,2], 
                                      adjusted.se.triple.1995.1999.so.cal[,2], 
                                      adjusted.se.did.1993.2009.so.cal[,2],
                                      adjusted.se.1993.2009.so.cal[,2], 
                                      adjusted.se.triple.1993.2009.so.cal[,2]),
                            p = list(adjusted.se.did.1995.1999.so.cal[,4],
                                     adjusted.se.1995.1999.so.cal[,4], 
                                     adjusted.se.triple.1995.1999.so.cal[,4], 
                                     adjusted.se.did.1993.2009.so.cal[,4],
                                     adjusted.se.1993.2009.so.cal[,4], 
                                     adjusted.se.triple.1993.2009.so.cal[,4]),
                            covariate.labels = cov.label,
                            title = title,
                            column.separate = c(1,1,1,1,1,1),
                            digits = 2,
                            dep.var.labels = "",
                            dep.var.caption = "",
                            notes.append = FALSE,
                            column.sep.width = "1pt",
                            no.space = TRUE,
                            type = "text",
                            omit.stat = c("rsq", "adj.rsq", "f"),
                            notes.align = "l",
                            add.lines = list(c("R2", 
                                               r2.did.1995.1999.so.cal,
                                               r2.1995.1999.so.cal, 
                                               r2.triple.1995.1999.so.cal,
                                               r2.did.1993.2009.so.cal,
                                               r2.1993.2009.so.cal,
                                               r2.triple.1993.2009.so.cal),
                                             c("F ratio", "", "", paste0(formatSig(f1[1], 2), ifelse(f1[2]<0.01, "***", 
                                                                                                  ifelse(f1[2]<0.05, "**", 
                                                                                                         ifelse(f1[2]<0.10, "*")))),
                                                    "","", paste0(formatSig(f2[1], 2), ifelse(f2[2]<0.01, "***", 
                                                                                           ifelse(f2[2]<0.05, "**", 
                                                                                                  ifelse(f2[2]<0.10, "*")))))))


