## ======================================
## Figure 4: Salmonella time series for reported illness rates (per 100,000)
## and hospitalization rates (per 100,000) (grey) for five Southern CA counties 
## and the rest of CA excluding those counties.
## ======================================

  rm(list=ls())
  
  #Assume base R is running
  my.wd <- getSrcDirectory(function(x){x})
  
  #Check for errors
  if(grepl("error", tolower(class(my.wd)[1])) | my.wd==""){
    #Try to access working directory through R Studio API
    my.wd <- tryCatch(dirname(rstudioapi::getActiveDocumentContext()$path),
                      error = function(e) e)
    
  }
  
  #Set working directory
  setwd(my.wd)

## ==========================
## Read in data
## ==========================  

  #County population
  county.pop.1964.2015 <- read.csv("../Data/county_population_1964_2015.csv", stringsAsFactors = FALSE)
  
  #Hospitalizations
  hosp.cty.yr.1983.2009 <- read.csv("../Data/hosp_county_year_1983_2009.csv", stringsAsFactors = FALSE)

  #Salmonella notifiables
  salm.notif.cty.yr.1964.2015 <- read.csv("../Data/illness_salm_1964_2015.csv", stringsAsFactors = FALSE)
  
## ======================================
## Summarize notifiable and hospitalization data
## ======================================
  
  #Years of data
  years <- 1964:2015
  hosp.years <- c(1983:2009)
                  
  #Identify name of salmonella column to use
  salm.col <- "salm"
  pop.col <- "pop"
  
  #Hospitalization time series
  la.series.hosp <- hosp.cty.yr.1983.2009[hosp.cty.yr.1983.2009$county.name=="Los Angeles", salm.col]
  oc.series.hosp <- hosp.cty.yr.1983.2009[hosp.cty.yr.1983.2009$county.name=="Orange", salm.col]
  rc.series.hosp <- hosp.cty.yr.1983.2009[hosp.cty.yr.1983.2009$county.name=="Riverside", salm.col]
  sb.series.hosp <- hosp.cty.yr.1983.2009[hosp.cty.yr.1983.2009$county.name=="San Bernardino", salm.col]
  sd.series.hosp <- hosp.cty.yr.1983.2009[hosp.cty.yr.1983.2009$county.name=="San Diego", salm.col]
  ca.series.hosp <- aggregate(formula(paste0(salm.col, "~year")), data = hosp.cty.yr.1983.2009, FUN = sum)[,2]
  ca.nola.series.hosp <- ca.series.hosp - la.series.hosp
  ca.series.hosp <- ca.series.hosp - la.series.hosp - oc.series.hosp - rc.series.hosp - sb.series.hosp - sd.series.hosp
  
  #Impute 1988 as the average of 1987 and 1989 for hospitalization time series
  series.list <- list(la.series.hosp, oc.series.hosp, rc.series.hosp, sb.series.hosp, sd.series.hosp, ca.series.hosp,ca.nola.series.hosp)
  names(series.list) <- c("la.series.hosp", "oc.series.hosp", "rc.series.hosp", "sb.series.hosp", "sd.series.hosp", "ca.series.hosp","ca.nola.series.hosp")
  for(i in 1:length(series.list)){
      series.list[[i]] <- c(series.list[[i]][hosp.years%in%1983:1987],
                            mean(series.list[[i]][hosp.years%in%c(1987,1989)]),
                            series.list[[i]][hosp.years%in%1989:2009])
  }
    
  #Notifiables time series
  la.series.notif <- as.vector(as.matrix(salm.notif.cty.yr.1964.2015[2,2:ncol(salm.notif.cty.yr.1964.2015)]))
  oc.series.notif <- as.vector(as.matrix(salm.notif.cty.yr.1964.2015[3,2:ncol(salm.notif.cty.yr.1964.2015)]))
  rc.series.notif <- as.vector(as.matrix(salm.notif.cty.yr.1964.2015[4,2:ncol(salm.notif.cty.yr.1964.2015)]))
  sb.series.notif <- as.vector(as.matrix(salm.notif.cty.yr.1964.2015[5,2:ncol(salm.notif.cty.yr.1964.2015)]))
  sd.series.notif <- as.vector(as.matrix(salm.notif.cty.yr.1964.2015[6,2:ncol(salm.notif.cty.yr.1964.2015)]))
  ca.series.notif <- as.vector(as.matrix(salm.notif.cty.yr.1964.2015[1,2:ncol(salm.notif.cty.yr.1964.2015)]))
  ca.series.notif <- ca.series.notif - la.series.notif - oc.series.notif - rc.series.notif - sb.series.notif - sd.series.notif
  
  ## Our best sense is that Riverside is a coding error, so impute extreme value with the mean of neighboring years
  rc.series.notif[11] <- mean(c(rc.series.notif[10], rc.series.notif[12]))
  
  #Get county populations
  la.pop <- county.pop.1964.2015[county.pop.1964.2015$year%in%years & county.pop.1964.2015$Name=="Los Angeles","pop"]
  oc.pop <- county.pop.1964.2015[county.pop.1964.2015$year%in%years & county.pop.1964.2015$Name=="Orange","pop"]
  rc.pop <- county.pop.1964.2015[county.pop.1964.2015$year%in%years & county.pop.1964.2015$Name=="Riverside","pop"]
  sb.pop <- county.pop.1964.2015[county.pop.1964.2015$year%in%years & county.pop.1964.2015$Name=="San Bernardino","pop"]
  sd.pop <- county.pop.1964.2015[county.pop.1964.2015$year%in%years & county.pop.1964.2015$Name=="San Diego","pop"]
  ca.pop <- aggregate(pop~year, data = county.pop.1964.2015[county.pop.1964.2015$year%in%years,], FUN = sum)[,2]
  ca.nola.pop <- ca.pop - la.pop
  ca.pop <- ca.pop - la.pop - oc.pop - rc.pop - sb.pop - sd.pop

## ======================================
## Output: Plot
## ======================================

  #Colors for plot
  la.color <- "#636363"
  ca.color <- "#bdbdbd"
  
  #Code for plot
  par(mfrow = c(2,3), mar=c(2.5,2.5,2,2.5), mgp=c(1.25,0.3,0), tcl=-0.3)
  
  #LA notif
  plot(years, 100000*la.series.notif/la.pop, type="n",col="red", lwd=1.5,main="Los Angeles",
       ylab="Illness", xlab="Year",
       ylim=c(7,26),xlim=c(1964,2015))
  rect(1995, par("usr")[3], 1999, par("usr")[4], col = rgb(0,0,0,0.15), border = NA)
  abline(v=1997.5,col="white")
  lines(years, 100000*la.series.notif/la.pop, lwd=1.5)
  text(2004, 24.5, "J&L\nwindow",col="darkgrey")
  text(1975, 15, "Illness")
  text(1990,  12, "Hosp.", col="darkgrey")

  # #LA hosp
  par(new = T)
  plot(hosp.years, 100000*series.list$la.series.hosp/la.pop[years%in%hosp.years], axes = F, xlab = NA, ylab=NA, type = "l", lwd = 1, lty = 1,ylim=c(0,8), col="darkgrey",xlim=c(1964,2015))
  axis(side = 4,col.axis="darkgrey",col.ticks="darkgrey")
  mtext(side = 4, line = 1.2, 'Hospitalization', cex = 0.7,col="darkgrey")
   
  #OC notif
  plot(years, 100000*oc.series.notif/oc.pop, type="n",
       main="Orange",
       xlab="", ylab="",
       ylim=c(5,26),xlim=c(1964,2015))
  rect(1995, par("usr")[3], 1999, par("usr")[4], col = 
         rgb(0,0,0,0.15), border = NA)
  abline(v=1997.5,col="white")
  lines(years, 100000*oc.series.notif/oc.pop,lwd=1.5)
  
  # #OC hosp
   par(new = T)
   plot(hosp.years, 100000*series.list$oc.series.hosp/oc.pop[years%in%hosp.years], axes = F, xlab = NA, ylab=NA, type = "l", lwd = 1, lty = 1,ylim=c(0,8), col="darkgrey",xlim=c(1964,2015))
   axis(side = 4,col.axis="darkgrey",col.ticks="darkgrey")
  
  #San Diego notif
  plot(years, 100000*sd.series.notif/sd.pop, type="n",
       main="San Diego",
       xlab="", ylab="",
       ylim=c(5,26),xlim=c(1964,2015))
  rect(1995, par("usr")[3], 1999, par("usr")[4], col = rgb(0,0,0,0.15), border = NA)
  abline(v=1997.5,col="white")
  lines(years, 100000*sd.series.notif/sd.pop,lwd=1.5)
  
    # #San Diego hosp
  par(new = T)
     plot(hosp.years, 100000*series.list$sd.series.hosp/sd.pop[years%in%hosp.years], axes = F, xlab = NA, ylab=NA, type = "l", lwd = 1, lty = 1,col="darkgrey",xlim=c(1964,2015),ylim=c(0,8))
  axis(side = 4,col.axis="darkgrey",col.ticks="darkgrey")

  #San Bernardino notif
  plot(years, 100000*sb.series.notif/sb.pop, type="n",
       main="San Bernardino",
       xlab="", ylab="",
       ylim=c(5,26),xlim=c(1964,2015))
  rect(1995, par("usr")[3], 1999, par("usr")[4], col = rgb(0,0,0,0.15), border = NA)
  abline(v=1997.5,col="white")
  lines(years, 100000*sb.series.notif/sb.pop,lwd=1.5)
  
  # #San Bernardino hosp
   par(new = T)
   plot(hosp.years, 100000*series.list$sb.series.hosp/sb.pop[years%in%hosp.years], axes = F, xlab = NA, ylab=NA, type = "l", lwd = 1, lty = 1,col="darkgrey", xlim=c(1964,2015),ylim=c(0,8))
   axis(side = 4,col.axis="darkgrey",col.ticks="darkgrey")

  #Riverside notif
  plot(years, 100000*rc.series.notif/rc.pop,
       main="Riverside",
       xlab="", ylab="",
       ylim=c(3,22),type="n")
  rect(1995, par("usr")[3], 1999, par("usr")[4], col = rgb(0,0,0,0.15), border = NA)
  abline(v=1997.5,col="white")
  lines(years, 100000*rc.series.notif/rc.pop,lwd=1.5)
  
  # #Riverside hosp
   par(new = T)
   plot(hosp.years, 100000*series.list$rc.series.hosp/rc.pop[years%in%hosp.years], axes = F, xlab = NA, ylab=NA, type = "l", lwd = 1, lty = 1,col="darkgrey", xlim=c(1964,2015),ylim=c(0,8))
   axis(side = 4,col.axis="darkgrey",col.ticks="darkgrey")

  #Rest of CA notif
  plot(years, 100000*ca.series.notif/ca.pop, type="n",col="red", main="Rest of California",
       ylab="", xlab="Year",
       ylim=c(5,26),xlim=c(1964,2015))
  rect(1995, par("usr")[3], 1999, par("usr")[4], col = rgb(0,0,0,0.15), border = NA)
  abline(v=1997.5,col="white")
  lines(years, 100000*ca.series.notif/ca.pop,lwd=1.5)
  
  # #Rest of CA pop
   par(new = T)
   plot(hosp.years, 100000*series.list$ca.series.hosp/ca.pop[years%in%hosp.years], axes = F, xlab = NA, ylab=NA, type = "l", lwd = 1.5, lty = 1,col="darkgrey", xlim=c(1964,2015), ylim=c(0,8))
   axis(side = 4,col.axis="darkgrey",col.ticks="darkgrey")
