classdef SteadyStateUpstreamRA < handle
    
    properties (GetAccess = public, SetAccess = public)
        calibration         % the calibration being used
        ssmean              % mean value MP
        sigma               % risk aversion
        delta               % capital share
        Delta               % price dispersion distorsion
        beta                % discount factor
        gamma               % inverse Frisch elasticity
        theta               % weight disutility of labor
        lambda              % fraction NOT adjusting prices
        epsilon             % elasticity of demand
        R                   % nominal interest rate
        w                   % wage
        q                   % asset price
        infl                % SS inflation
        p_upstr             % Upstream price 
        N                   % N
        
        frac_cons=-1       % fraction of borrowing constrained using MC
    end
    
    properties (Access=protected)   % play with those if things don't work...
        verbose   = true            % switch to enable debug output
        tol = 1e-5;                 % max relative deviation for convergence
        weight_new = 1; 
    end
    
    properties (Dependent=true, SetAccess=protected)% play with those if things don't work...
        Y                   % Aggregate Output
        C                   % Aggregate Consumption
        A                   % Aggregate Savings
    end
    
    methods
        
        function self = SteadyStateUpstreamRA(calibr)
            % Constructor. pass it a calibration structure
            self.calibration = calibr;
            self.ssmean = calibr.ssmean;
            self.sigma = calibr.sigma;
            self.delta = calibr.delta;
            self.gamma = calibr.gamma;
            self.theta = calibr.theta;
            self.lambda = calibr.lambda;
            self.epsilon = calibr.epsilon;
            self.R = calibr.R;
            self.infl = calibr.infl;
            if self.lambda ~= 1
                self.Delta = (1-self.lambda)*((1-self.lambda*self.infl^(self.epsilon-1))/(1-self.lambda))^(self.epsilon/(self.epsilon-1))/(1-self.lambda*self.infl^self.epsilon);
            else
                self.Delta = 1; % Rigid prices
            end
			
            self.calculate_steady_state();
        end
        
        function res = get.Y(self)
            res = self.N^(1-self.delta)/self.Delta;
        end
        
        function res = get.C(self)
            res = self.Y;
        end
        
        function res = get.A(self)
            res = self.q*self.R;
        end
        
    end
    
    methods(Access = protected)
        
        function calculate_steady_state(self)
            self.beta = self.infl/self.R;
            self.p_upstr = (self.epsilon - 1)/self.epsilon;
            self.N = (self.p_upstr*(1-self.delta)*self.ssmean^(1+self.gamma)*self.Delta)^(1/(1+self.gamma+(self.sigma-1)*(1-self.delta)));
            
            self.w = self.p_upstr*(1 - self.delta)*self.N^(-self.delta);
            self.q = self.delta*self.Y/(self.R-1);
        end
                
    end
    
end